{$IFNDEF FPC_DOTTEDUNITS}
unit dbf_collate;
{$ENDIF FPC_DOTTEDUNITS}
{
    This file is part of the Free Pascal run time library.
    Copyright (c) 1999-2022 by Pascal Ganaye,Micha Nelissen and other members of the
    Free Pascal development team

    DBF  collation implementation

    See the file COPYING.FPC, included in this distribution,
    for details about the copyright.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

 **********************************************************************}
{$i dbf_common.inc}


interface

{$IFDEF FPC_DOTTEDUNITS}
uses 
  System.SysUtils;
{$ELSE FPC_DOTTEDUNITS}
uses 
  SysUtils;
{$ENDIF FPC_DOTTEDUNITS}



type
  TCollationTable         = array[0..255] of Byte;
  PCollationTable         = ^TCollationTable;
  TCollationBDEName       = string[8];

  TCollationDefinition = record
    CollationBDEName :TCollationBDEName;
    CollationTable   :PCollationTable;
  end;



const
  BINARY_COLLATION  :PCollationTable = PCollationTable($111);
  UNKNOWN_COLLATION :PCollationTable = PCollationTable($222);



function GetCollationTable( DbfLangId :integer ) :PCollationTable;
function DbfCompareString( CollationTable :PCollationTable; String1 :PAnsiChar; nLength1 :integer; String2 :PAnsiChar; nLength2 :integer ) :integer;
function RegisterCollation( DbfLangId :integer; CollationTable :PCollationTable; BDEName :TCollationBDEName ) :Boolean;


var
  CollationTables :array[0..255] of TCollationDefinition;


implementation
{$IFDEF FPC_DOTTEDUNITS}
uses Data.Dbf.Lang, System.Math;
{$ELSE FPC_DOTTEDUNITS}
uses dbf_lang, math;
{$ENDIF FPC_DOTTEDUNITS}


procedure InitialiseCollationTables;
var
  nCnt :integer;
begin
  for nCnt := Low(CollationTables) to High(CollationTables) do
    begin
      CollationTables[nCnt].CollationTable   := BINARY_COLLATION;
      CollationTables[nCnt].CollationBDEName := '';
    end;
end;

function GetCollationTable( DbfLangId :integer ) :PCollationTable;
begin
  if (DbfLangId < Low(CollationTables)) or (DbfLangId > High(CollationTables)) then
    begin
      Result := nil;
    end
  else
    begin
      Result := CollationTables[DbfLangId].CollationTable;
    end;
end;

function DbfCompareString( CollationTable :PCollationTable; String1 :PAnsiChar; nLength1 :integer; String2 :PAnsiChar; nLength2 :integer ) :integer;
var
  nCnt, nMax, nVal1, nVal2 :integer;
const
  ONE_LESS_THAN_TWO  = 1;
  EQUAL              = 2;
  TWO_LESS_THAN_ONE  = 3;
begin
  result := EQUAL;

  if CollationTable = BINARY_COLLATION then
    begin
      // binary collation
      nMax := Min( nLength1, nLength2 );
      for nCnt := 1 to nMax do
        begin
          if Ord(String1^) < Ord(String2^) then
            begin
              result := ONE_LESS_THAN_TWO;
              break;
            end
          else if Ord(String1^) > Ord(String2^) then
            begin
              result := TWO_LESS_THAN_ONE;
              break;
            end;

          Inc(String1);
          Inc(String2);
        end;
    end

  else

    begin
      // collation via collation table
      nMax := Min( nLength1, nLength2 );
      for nCnt := 1 to nMax do
        begin
          nVal1 := CollationTable[Ord(String1^)];
          nVal2 := CollationTable[Ord(String2^)];

          if nVal1 < nVal2 then
            begin
              result := ONE_LESS_THAN_TWO;
              break;
            end

          else if nVal1 > nVal2 then
            begin
              result := TWO_LESS_THAN_ONE;
              break;
            end;

          Inc(String1);
          Inc(String2);
        end;

    end;


  if result = EQUAL then
    begin
      // If the two strings are of different lengths, they are compared up to the
      // length of the shortest one. If they are equal to that point, then the
      // return value will indicate that the longer string is greater.
      if nLength1 < nLength2 then
        result := ONE_LESS_THAN_TWO
      else if nLength1 > nLength2 then
        result := TWO_LESS_THAN_ONE;
    end;

end;


function RegisterCollation( DbfLangId :integer; CollationTable :PCollationTable; BDEName :TCollationBDEName ) :Boolean;
begin
  if (DbfLangId < Low(CollationTables)) or (DbfLangId > High(CollationTables)) then
    begin
      result := false;
    end
  else
    begin
      CollationTables[DbfLangId].CollationTable   := CollationTable;
      CollationTables[DbfLangId].CollationBDEName := BDEName;
      result := true;
    end;
end;




const

  {$IFDEF PARADOX_COLLATIONS}

  // intl    55764
  _intl :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 069, 073, 076, 079, 085, 088, 090, 092, 098, 100, 102, 104, 106, 109, 115,
    119, 121, 123, 125, 128, 133, 137, 139, 141, 144, 146, 149, 150, 151, 152, 153,
    154, 065, 072, 074, 078, 080, 087, 089, 091, 093, 099, 101, 103, 105, 107, 111,
    118, 120, 122, 124, 127, 129, 136, 138, 140, 142, 145, 155, 156, 157, 158, 159,
    035, 035, 035, 165, 035, 035, 035, 035, 035, 035, 125, 035, 115, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 124, 035, 111, 035, 035, 144,
    181, 172, 162, 163, 035, 164, 035, 035, 035, 035, 166, 173, 169, 035, 035, 035,
    178, 176, 180, 035, 035, 175, 035, 179, 035, 035, 167, 174, 171, 170, 035, 168,
    069, 069, 069, 069, 071, 148, 161, 077, 085, 086, 085, 085, 098, 098, 098, 098, 
    079, 110, 115, 115, 115, 115, 117, 035, 115, 133, 133, 133, 135, 144, 035, 126, 
    067, 066, 068, 065, 070, 147, 160, 075, 083, 082, 084, 081, 096, 095, 097, 094, 
    078, 108, 113, 112, 114, 111, 116, 177, 111, 131, 130, 132, 134, 142, 035, 143
  );

  intl :PCollationTable = @_intl;

  {$ENDIF}




  {$IFDEF PARADOX_COLLATIONS}

  // nordan    1610

  _nordan :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 069, 071, 074, 077, 083, 086, 088, 090, 096, 098, 100, 102, 104, 107, 113,
    115, 117, 119, 121, 123, 128, 130, 132, 134, 138, 141, 152, 153, 154, 155, 156,
    157, 065, 070, 072, 076, 078, 085, 087, 089, 091, 097, 099, 101, 103, 105, 109,
    114, 116, 118, 120, 122, 124, 129, 131, 133, 135, 140, 158, 159, 160, 161, 162,
    035, 035, 035, 164, 035, 035, 035, 035, 035, 035, 121, 035, 113, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 120, 035, 109, 035, 035, 138,
    181, 171, 035, 163, 173, 035, 035, 035, 035, 035, 165, 172, 168, 035, 035, 035,
    178, 176, 180, 035, 035, 175, 035, 179, 035, 035, 166, 035, 170, 169, 035, 167,
    069, 069, 069, 069, 145, 151, 144, 075, 083, 084, 083, 083, 096, 096, 096, 096,
    077, 108, 113, 113, 113, 113, 149, 035, 148, 128, 128, 128, 139, 138, 035, 174,
    067, 066, 068, 065, 143, 150, 142, 073, 081, 080, 082, 079, 094, 093, 095, 092,
    076, 106, 111, 110, 112, 109, 147, 177, 146, 126, 125, 127, 136, 135, 035, 137
  );
  nordan :PCollationTable = @_nordan;

  {$ENDIF}




  {$IFDEF PARADOX_COLLATIONS}

  // nordan40    31338

  _nordan40 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 065, 070, 072, 076, 078, 085, 087, 089, 091, 097, 099, 101, 103, 105, 109,
    114, 116, 118, 120, 122, 124, 129, 131, 133, 135, 140, 152, 153, 154, 155, 156,
    157, 066, 071, 074, 077, 080, 086, 088, 090, 092, 098, 100, 102, 104, 107, 110,
    115, 117, 119, 121, 123, 125, 130, 132, 134, 137, 141, 158, 159, 160, 161, 162,
    035, 035, 035, 164, 035, 035, 035, 035, 035, 035, 120, 035, 109, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 121, 035, 110, 035, 035, 135,
    181, 171, 035, 163, 173, 035, 035, 035, 035, 035, 165, 172, 168, 035, 035, 035,
    178, 176, 180, 035, 035, 175, 035, 179, 035, 035, 166, 035, 170, 169, 035, 167,
    065, 065, 065, 065, 143, 150, 142, 073, 078, 079, 078, 078, 091, 091, 091, 091,
    076, 106, 109, 109, 109, 109, 147, 035, 146, 124, 124, 124, 136, 135, 035, 174,
    068, 067, 069, 066, 145, 151, 144, 075, 083, 082, 084, 081, 095, 094, 096, 093,
    077, 108, 112, 111, 113, 110, 149, 177, 148, 127, 126, 128, 139, 137, 035, 138
  );
  nordan40 :PCollationTable = @_nordan40;

  {$ENDIF}




  {$IFDEF PARADOX_COLLATIONS}

  // swedfin    6461

  _swedfin :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 069, 071, 074, 077, 083, 086, 088, 090, 096, 098, 100, 102, 104, 107, 113,
    115, 117, 119, 121, 123, 128, 130, 132, 134, 137, 139, 150, 151, 152, 153, 154,
    155, 065, 070, 072, 076, 078, 085, 087, 089, 091, 097, 099, 101, 103, 105, 109,
    114, 116, 118, 120, 122, 124, 129, 131, 133, 135, 138, 156, 157, 158, 159, 160,
    035, 035, 035, 164, 035, 035, 035, 035, 035, 035, 121, 035, 113, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 120, 035, 109, 035, 035, 137,
    181, 171, 161, 162, 035, 163, 035, 035, 035, 035, 165, 172, 168, 035, 035, 035,
    178, 176, 180, 035, 035, 175, 035, 179, 035, 035, 166, 173, 170, 169, 035, 167,
    069, 069, 069, 069, 145, 143, 141, 075, 083, 084, 083, 083, 096, 096, 096, 096,
    077, 108, 113, 113, 113, 113, 147, 035, 113, 128, 128, 128, 149, 137, 035, 174,
    067, 066, 068, 065, 144, 142, 140, 073, 081, 080, 082, 079, 094, 093, 095, 092,
    076, 106, 111, 110, 112, 109, 146, 177, 109, 126, 125, 127, 148, 135, 035, 136
  );
  swedfin :PCollationTable = @_swedfin;

  {$ENDIF}




  // DB865DA0      Checksum: 51597

  _DB865DA0 :TCollationTable = (
    000, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 
    153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 
    001, 118, 130, 124, 125, 110, 123, 132, 099, 100, 111, 108, 113, 109, 112, 119, 
    002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 115, 114, 105, 107, 106, 117, 
    122, 012, 013, 014, 016, 017, 019, 020, 021, 022, 023, 024, 025, 026, 027, 029, 
    030, 031, 032, 033, 034, 035, 037, 038, 039, 040, 041, 101, 120, 102, 133, 135, 
    131, 047, 051, 052, 054, 055, 060, 061, 062, 063, 068, 069, 070, 071, 072, 074, 
    078, 079, 080, 081, 082, 083, 088, 089, 090, 091, 093, 103, 121, 104, 134, 169, 
    015, 087, 056, 050, 095, 049, 098, 053, 058, 059, 057, 067, 066, 065, 043, 046, 
    018, 094, 042, 077, 097, 076, 086, 085, 092, 045, 036, 096, 126, 044, 128, 127, 
    048, 064, 075, 084, 073, 028, 136, 137, 116, 170, 171, 172, 173, 174, 175, 176, 
    177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 
    193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 
    209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 
    225, 129, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  DB865DA0 :PCollationTable = @_DB865DA0;

  DB865NO0 :PCollationTable = @_DB865DA0;




  // DB437NL0      Checksum: 26538

  _DB437NL0 :TCollationTable = (
    000, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 
    153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 
    001, 114, 130, 120, 121, 106, 119, 132, 095, 096, 107, 104, 109, 105, 108, 115, 
    002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 111, 110, 101, 103, 102, 113, 
    118, 012, 015, 016, 018, 019, 021, 022, 023, 024, 025, 026, 027, 028, 029, 031, 
    033, 034, 035, 036, 037, 038, 040, 041, 042, 043, 044, 097, 116, 098, 133, 135, 
    131, 045, 051, 052, 054, 055, 060, 061, 062, 063, 068, 069, 070, 071, 072, 074, 
    079, 080, 081, 082, 083, 084, 089, 090, 091, 092, 094, 099, 117, 100, 134, 169, 
    017, 088, 056, 048, 049, 047, 050, 053, 058, 059, 057, 067, 066, 065, 013, 014, 
    020, 127, 128, 077, 078, 076, 087, 086, 093, 032, 039, 122, 123, 124, 126, 125, 
    046, 064, 075, 085, 073, 030, 136, 137, 112, 170, 171, 172, 173, 174, 175, 176, 
    177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 
    193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 
    209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 
    225, 129, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  DB437NL0 :PCollationTable = @_DB437NL0;

  DB437UK0 :PCollationTable = @_DB437NL0;

  DB437IT0 :PCollationTable = @_DB437NL0;

  DB437FR0 :PCollationTable = @_DB437NL0;




  // DB850NL0      Checksum: 4071

  _DB850NL0 :TCollationTable = (
    000, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 
    155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 
    001, 176, 189, 213, 206, 214, 212, 006, 192, 193, 210, 215, 173, 005, 181, 180, 
    007, 011, 013, 015, 017, 018, 019, 020, 021, 022, 175, 174, 219, 220, 221, 178, 
    203, 024, 041, 043, 047, 051, 061, 063, 065, 067, 078, 080, 082, 084, 086, 090, 
    105, 107, 109, 111, 114, 118, 128, 130, 132, 134, 139, 194, 211, 195, 184, 003, 
    183, 023, 040, 042, 046, 050, 060, 062, 064, 066, 077, 079, 081, 083, 085, 089, 
    104, 106, 108, 110, 113, 117, 127, 129, 131, 133, 138, 196, 223, 197, 186, 227, 
    045, 125, 052, 030, 034, 028, 032, 044, 056, 058, 054, 074, 072, 070, 035, 033, 
    053, 038, 039, 096, 098, 094, 123, 121, 137, 099, 126, 102, 208, 103, 218, 207, 
    026, 068, 092, 119, 087, 088, 025, 091, 179, 202, 222, 009, 008, 177, 190, 191, 
    230, 231, 232, 243, 239, 027, 031, 029, 201, 250, 254, 247, 253, 205, 209, 236, 
    240, 241, 235, 237, 244, 238, 036, 037, 251, 245, 252, 246, 248, 255, 249, 204, 
    048, 049, 057, 059, 055, 076, 069, 073, 075, 242, 234, 233, 229, 224, 071, 228, 
    093, 112, 097, 095, 100, 101, 226, 115, 116, 120, 124, 122, 135, 136, 171, 182, 
    004, 216, 172, 010, 200, 199, 217, 188, 225, 185, 187, 012, 016, 014, 198, 002
  );
  DB850NL0 :PCollationTable = @_DB850NL0;

  DB850FR0 :PCollationTable = @_DB850NL0;

  DB850IT1 :PCollationTable = @_DB850NL0;

  DB850ES0 :PCollationTable = @_DB850NL0;

  DB850UK0 :PCollationTable = @_DB850NL0;




  // DB437FI0      Checksum: 33363

  _DB437FI0 :TCollationTable = (
    000, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 
    153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 
    001, 118, 130, 124, 125, 110, 123, 132, 099, 100, 111, 108, 113, 109, 112, 119, 
    002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 115, 114, 105, 107, 106, 117, 
    122, 012, 013, 014, 016, 017, 019, 020, 021, 022, 023, 024, 025, 026, 027, 029, 
    030, 031, 032, 033, 034, 035, 037, 038, 039, 040, 041, 101, 120, 102, 133, 135, 
    131, 047, 051, 052, 054, 055, 060, 061, 062, 063, 068, 069, 070, 071, 072, 074, 
    078, 079, 080, 081, 082, 083, 088, 089, 090, 091, 093, 103, 121, 104, 134, 169, 
    015, 087, 056, 050, 095, 049, 094, 053, 058, 059, 057, 067, 066, 065, 043, 042, 
    018, 096, 044, 077, 097, 076, 086, 085, 092, 045, 036, 098, 126, 046, 128, 127, 
    048, 064, 075, 084, 073, 028, 136, 137, 116, 170, 171, 172, 173, 174, 175, 176, 
    177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 
    193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 
    209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 
    225, 129, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  DB437FI0 :PCollationTable = @_DB437FI0;




  // DB437DE0      Checksum: 51657
  _DB437DE0 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 
    158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 048, 049, 050, 051, 052, 053, 
    054, 168, 172, 173, 175, 176, 178, 179, 180, 181, 182, 183, 184, 185, 186, 188, 
    190, 191, 192, 193, 194, 195, 197, 198, 199, 200, 201, 055, 056, 057, 058, 059, 
    060, 202, 210, 211, 213, 214, 219, 220, 221, 222, 227, 228, 229, 230, 231, 233, 
    239, 240, 241, 242, 244, 245, 250, 251, 252, 253, 255, 061, 062, 063, 064, 065, 
    174, 249, 215, 205, 206, 204, 207, 212, 217, 218, 216, 226, 225, 224, 169, 170, 
    177, 208, 171, 236, 237, 235, 248, 247, 254, 189, 196, 066, 067, 068, 069, 070, 
    203, 223, 234, 246, 232, 187, 209, 238, 071, 072, 073, 074, 075, 076, 077, 078, 
    079, 080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 090, 091, 092, 093, 094, 
    095, 096, 097, 098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 
    111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 
    127, 243, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 
    142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157
  );

  DB437DE0 :PCollationTable = @_DB437DE0;




  // DB850DE0      Checksum: 60384
  _DB850DE0 :TCollationTable = (
    000, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016, 017, 018, 019, 020, 021, 
    022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 032, 033, 034, 035, 036, 037, 
    001, 043, 056, 080, 073, 081, 079, 006, 059, 060, 077, 082, 040, 005, 048, 047, 
    240, 244, 246, 248, 250, 251, 252, 253, 254, 255, 042, 041, 086, 087, 088, 045, 
    070, 124, 141, 143, 147, 151, 161, 163, 165, 167, 178, 180, 182, 184, 186, 190, 
    205, 207, 209, 211, 214, 218, 228, 230, 232, 234, 239, 061, 078, 062, 051, 003, 
    050, 123, 140, 142, 146, 150, 160, 162, 164, 166, 177, 179, 181, 183, 185, 189, 
    204, 206, 208, 210, 213, 217, 227, 229, 231, 233, 238, 063, 090, 064, 053, 094, 
    145, 225, 152, 130, 134, 128, 132, 144, 156, 158, 154, 174, 172, 170, 135, 133, 
    153, 138, 139, 196, 198, 194, 223, 221, 237, 199, 226, 202, 075, 203, 085, 074, 
    126, 168, 192, 219, 187, 188, 125, 191, 046, 069, 089, 242, 241, 044, 057, 058, 
    097, 098, 099, 110, 106, 127, 131, 129, 068, 117, 121, 114, 120, 072, 076, 103, 
    107, 108, 102, 104, 111, 105, 136, 137, 118, 112, 119, 113, 115, 122, 116, 071, 
    148, 149, 157, 159, 155, 176, 169, 173, 175, 109, 101, 100, 096, 091, 171, 095, 
    193, 212, 197, 195, 200, 201, 093, 215, 216, 220, 224, 222, 235, 236, 038, 049, 
    004, 083, 039, 243, 067, 066, 084, 055, 092, 052, 054, 245, 249, 247, 065, 002
  );

  DB850DE0 :PCollationTable = @_DB850DE0;




  // DB437SV0      Checksum: 5835

  _DB437SV0 :TCollationTable = (
    000, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 
    153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 
    001, 117, 130, 123, 124, 109, 122, 132, 098, 099, 110, 107, 112, 108, 111, 118, 
    002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 114, 113, 104, 106, 105, 116, 
    121, 012, 013, 014, 016, 017, 019, 020, 021, 022, 023, 024, 025, 026, 027, 029, 
    030, 031, 032, 033, 034, 035, 037, 038, 039, 040, 041, 100, 119, 101, 133, 135, 
    131, 045, 049, 050, 052, 053, 058, 059, 060, 061, 066, 067, 068, 069, 070, 072, 
    076, 077, 078, 079, 081, 082, 087, 088, 089, 090, 092, 102, 120, 103, 134, 169, 
    015, 086, 054, 048, 094, 047, 093, 051, 056, 057, 055, 065, 064, 063, 043, 042, 
    018, 095, 096, 075, 097, 074, 085, 084, 091, 044, 036, 125, 126, 127, 129, 128, 
    046, 062, 073, 083, 071, 028, 136, 137, 115, 170, 171, 172, 173, 174, 175, 176, 
    177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 
    193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 
    209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 
    225, 080, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  DB437SV0 :PCollationTable = @_DB437SV0;




  // DB850SV1      Checksum: 15648

  _DB850SV1 :TCollationTable = (
    000, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 
    155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 
    001, 176, 189, 213, 206, 214, 212, 006, 192, 193, 210, 215, 173, 005, 181, 180, 
    007, 011, 013, 015, 017, 018, 019, 020, 021, 022, 175, 174, 219, 220, 221, 178, 
    203, 024, 035, 037, 041, 045, 055, 057, 059, 061, 072, 074, 076, 078, 080, 084, 
    095, 097, 099, 101, 104, 108, 116, 118, 120, 122, 129, 194, 211, 195, 184, 003, 
    183, 023, 034, 036, 040, 044, 054, 056, 058, 060, 071, 073, 075, 077, 079, 083, 
    094, 096, 098, 100, 103, 107, 115, 117, 119, 121, 128, 196, 223, 197, 186, 227, 
    039, 126, 046, 030, 132, 028, 130, 038, 050, 052, 048, 068, 066, 064, 133, 131, 
    047, 134, 135, 090, 136, 088, 113, 111, 125, 137, 127, 138, 208, 139, 218, 207, 
    026, 062, 086, 109, 081, 082, 025, 085, 179, 202, 222, 009, 008, 177, 190, 191, 
    230, 231, 232, 243, 239, 027, 031, 029, 201, 250, 254, 247, 253, 205, 209, 236, 
    240, 241, 235, 237, 244, 238, 032, 033, 251, 245, 252, 246, 248, 255, 249, 204, 
    042, 043, 051, 053, 049, 070, 063, 067, 069, 242, 234, 233, 229, 224, 065, 228, 
    087, 102, 091, 089, 092, 093, 226, 105, 106, 110, 114, 112, 123, 124, 171, 182, 
    004, 216, 172, 010, 200, 199, 217, 188, 225, 185, 187, 012, 016, 014, 198, 002
  );
  DB850SV1 :PCollationTable = @_DB850SV1;




  // DB437ES1      Checksum: 29895

  _DB437ES1 :TCollationTable = (
    000, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 
    154, 155, 156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 
    001, 114, 131, 121, 122, 107, 120, 133, 096, 097, 108, 105, 002, 106, 109, 116, 
    003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 111, 110, 102, 104, 103, 113, 
    119, 013, 016, 017, 019, 020, 022, 023, 024, 025, 026, 027, 028, 029, 030, 032, 
    034, 035, 036, 037, 038, 039, 041, 042, 043, 044, 045, 098, 117, 099, 134, 136, 
    132, 046, 052, 053, 055, 056, 061, 062, 063, 064, 069, 070, 071, 072, 073, 075, 
    080, 081, 082, 083, 084, 085, 090, 091, 092, 093, 095, 100, 118, 101, 135, 170, 
    018, 088, 057, 049, 050, 048, 051, 054, 059, 060, 058, 067, 068, 066, 014, 015, 
    021, 128, 129, 078, 079, 077, 089, 087, 094, 033, 040, 123, 124, 125, 126, 127, 
    047, 065, 076, 086, 074, 031, 137, 138, 112, 171, 172, 173, 174, 115, 175, 176, 
    177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 
    193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 
    209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 
    225, 130, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  DB437ES1 :PCollationTable = @_DB437ES1 ;




  // DB437US0      Checksum: 43413

  _DB437US0 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 
    064, 065, 066, 067, 068, 069, 070, 071, 072, 073, 074, 075, 076, 077, 078, 079, 
    080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 090, 091, 092, 093, 094, 095, 
    096, 097, 098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 
    112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 
    128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 
    144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 
    160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 
    176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 
    192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 
    208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 
    224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  DB437US0 :PCollationTable = @_DB437US0;




  // DB863CF1      Checksum: 34446

  _DB863CF1 :TCollationTable = (
    000, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 
    146, 147, 148, 149, 150, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 
    001, 110, 123, 116, 117, 103, 115, 125, 092, 093, 104, 101, 106, 102, 105, 111, 
    002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 108, 107, 098, 100, 099, 109, 
    114, 012, 015, 016, 018, 019, 024, 025, 026, 027, 030, 031, 032, 033, 034, 035, 
    037, 038, 039, 040, 041, 042, 046, 047, 048, 049, 050, 094, 112, 095, 126, 128, 
    124, 051, 054, 055, 057, 058, 063, 064, 065, 066, 069, 070, 071, 072, 073, 074, 
    077, 078, 079, 080, 081, 082, 087, 088, 089, 090, 091, 096, 113, 097, 127, 163, 
    017, 086, 059, 053, 014, 052, 151, 056, 061, 062, 060, 068, 067, 252, 013, 121, 
    020, 021, 022, 076, 023, 029, 085, 084, 254, 036, 045, 118, 119, 043, 044, 120, 
    250, 251, 075, 083, 253, 249, 129, 130, 028, 164, 165, 166, 167, 168, 169, 170, 
    171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 
    187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 
    203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 
    219, 122, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 
    234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 255
  );
  DB863CF1 :PCollationTable = @_DB863CF1;




  // DB850CF0      Checksum: 4704

  _DB850CF0 :TCollationTable = (
    000, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 
    175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 
    001, 136, 152, 142, 143, 128, 141, 154, 117, 118, 129, 126, 131, 127, 130, 137, 
    002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 133, 132, 123, 125, 124, 135, 
    140, 012, 019, 020, 022, 023, 028, 029, 030, 031, 036, 037, 038, 039, 040, 042, 
    048, 049, 050, 051, 052, 053, 058, 059, 060, 061, 063, 119, 138, 120, 155, 157, 
    153, 064, 071, 072, 074, 075, 080, 081, 082, 083, 088, 089, 090, 091, 092, 094, 
    100, 101, 102, 103, 104, 105, 110, 111, 112, 113, 116, 121, 139, 122, 156, 191, 
    021, 109, 076, 067, 068, 066, 069, 073, 078, 079, 077, 087, 086, 085, 016, 017, 
    024, 149, 150, 097, 098, 096, 108, 107, 115, 046, 057, 144, 145, 146, 148, 147, 
    065, 084, 095, 106, 093, 041, 158, 159, 134, 192, 193, 194, 195, 196, 197, 198, 
    199, 200, 201, 202, 203, 013, 015, 014, 204, 205, 206, 207, 208, 209, 210, 211, 
    212, 213, 214, 215, 216, 217, 070, 018, 218, 219, 220, 221, 222, 223, 224, 225, 
    226, 227, 026, 027, 025, 228, 032, 034, 035, 229, 230, 231, 232, 233, 033, 234, 
    043, 151, 045, 044, 099, 047, 235, 236, 237, 054, 056, 055, 114, 062, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  DB850CF0 :PCollationTable = @_DB850CF0;




  // DB852CZ0      Checksum: 35661

  _DB852CZ0 :TCollationTable = (
    000, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 172, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 
    001, 149, 151, 192, 174, 173, 169, 152, 159, 160, 191, 187, 146, 154, 145, 157, 
    135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 150, 147, 163, 186, 164, 148, 
    170, 003, 015, 017, 025, 031, 041, 043, 045, 047, 053, 055, 057, 065, 067, 073, 
    083, 085, 087, 094, 102, 108, 118, 120, 122, 124, 128, 161, 158, 162, 188, 155, 
    153, 002, 014, 016, 024, 030, 040, 042, 044, 046, 052, 054, 056, 064, 066, 072, 
    082, 084, 086, 093, 101, 107, 117, 119, 121, 123, 127, 165, 156, 166, 193, 222, 
    021, 113, 032, 006, 010, 111, 018, 020, 062, 036, 081, 080, 050, 132, 011, 019, 
    033, 059, 058, 076, 078, 061, 060, 096, 095, 079, 114, 104, 103, 063, 189, 022, 
    004, 048, 074, 109, 013, 012, 134, 133, 039, 038, 255, 131, 023, 097, 167, 168, 
    216, 217, 218, 194, 195, 005, 007, 035, 098, 205, 206, 207, 208, 130, 129, 196, 
    197, 198, 199, 200, 201, 202, 009, 008, 209, 210, 211, 212, 213, 214, 215, 175, 
    028, 029, 027, 037, 026, 071, 049, 051, 034, 203, 204, 219, 220, 106, 112, 221, 
    075, 092, 077, 069, 068, 070, 100, 099, 089, 110, 088, 116, 125, 126, 105, 177, 
    185, 182, 184, 178, 179, 171, 190, 183, 180, 181, 176, 115, 091, 090, 223, 224
  );
  DB852CZ0 :PCollationTable = @_DB852CZ0;




  // DB867CZ0      Checksum: 9099

  _DB867CZ0 :TCollationTable = (
    000, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255, 
    001, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 
    099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 124, 125, 126, 127, 128, 129, 
    130, 003, 009, 011, 015, 019, 025, 027, 029, 031, 035, 037, 039, 045, 047, 051, 
    059, 061, 063, 070, 074, 078, 086, 088, 090, 092, 096, 131, 132, 133, 134, 135, 
    136, 002, 008, 010, 014, 018, 024, 026, 028, 030, 034, 036, 038, 044, 046, 050, 
    058, 060, 062, 069, 073, 077, 085, 087, 089, 091, 095, 137, 138, 139, 140, 141, 
    013, 083, 020, 016, 006, 017, 076, 012, 022, 023, 041, 033, 042, 040, 007, 005, 
    021, 097, 098, 054, 056, 053, 081, 080, 093, 057, 084, 072, 043, 094, 067, 075, 
    004, 032, 052, 079, 048, 049, 082, 055, 071, 066, 064, 065, 142, 143, 144, 145, 
    146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161, 
    162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 
    178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 
    194, 068, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 
    209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224
  );
  DB867CZ0 :PCollationTable = @_DB867CZ0;




  // db852hdc      Checksum: 7023

  _db852hdc :TCollationTable = (
    000, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 
    193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 
    001, 162, 172, 168, 169, 155, 167, 174, 144, 145, 156, 153, 158, 154, 157, 163, 
    002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 160, 159, 150, 152, 151, 161, 
    166, 012, 018, 019, 023, 026, 031, 032, 033, 034, 038, 039, 040, 044, 045, 048, 
    053, 054, 055, 058, 062, 065, 070, 071, 072, 073, 075, 146, 164, 147, 175, 177, 
    173, 079, 085, 086, 090, 093, 098, 099, 100, 101, 104, 105, 106, 109, 110, 113, 
    118, 119, 120, 123, 127, 130, 135, 136, 137, 138, 140, 148, 165, 149, 176, 209, 
    020, 133, 097, 082, 080, 131, 089, 087, 107, 094, 052, 117, 102, 078, 013, 022, 
    030, 043, 036, 114, 116, 042, 108, 061, 126, 051, 068, 064, 129, 041, 170, 088, 
    084, 103, 115, 132, 014, 081, 077, 142, 028, 095, 210, 143, 021, 124, 211, 212, 
    213, 214, 215, 216, 217, 017, 015, 029, 059, 218, 219, 220, 221, 076, 141, 222, 
    223, 224, 225, 226, 227, 228, 016, 083, 229, 230, 231, 232, 233, 234, 235, 236, 
    091, 024, 025, 027, 092, 047, 037, 035, 096, 237, 238, 239, 240, 063, 066, 241, 
    050, 171, 049, 046, 112, 111, 060, 125, 057, 067, 122, 069, 139, 074, 128, 242, 
    243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 134, 056, 121, 254, 255
  );
  db852hdc :PCollationTable = @_db852hdc;




  // db852po0      Checksum: 64062

  _db852po0 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 
    032, 194, 203, 200, 201, 187, 199, 205, 176, 177, 188, 185, 190, 186, 189, 195, 
    033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 192, 191, 182, 184, 183, 193, 
    198, 044, 056, 058, 066, 072, 082, 084, 086, 088, 094, 096, 098, 106, 108, 114, 
    124, 126, 128, 134, 143, 149, 159, 161, 163, 165, 169, 178, 196, 179, 206, 208, 
    204, 043, 055, 057, 065, 071, 081, 083, 085, 087, 093, 095, 097, 105, 107, 113, 
    123, 125, 127, 133, 142, 148, 158, 160, 162, 164, 168, 180, 197, 181, 207, 209, 
    062, 154, 073, 047, 051, 152, 059, 061, 103, 077, 122, 121, 091, 171, 052, 060, 
    074, 100, 099, 117, 119, 102, 101, 139, 138, 120, 155, 145, 144, 104, 202, 063, 
    045, 089, 115, 150, 053, 054, 175, 174, 080, 079, 210, 170, 064, 135, 211, 212, 
    213, 214, 215, 216, 217, 046, 050, 076, 136, 218, 219, 220, 221, 173, 172, 222, 
    223, 224, 225, 226, 227, 228, 049, 048, 229, 230, 231, 232, 233, 234, 235, 236, 
    069, 070, 068, 078, 067, 112, 090, 092, 075, 237, 238, 239, 240, 147, 153, 241, 
    116, 137, 118, 110, 109, 111, 141, 140, 130, 151, 129, 157, 166, 167, 146, 242, 
    243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 156, 132, 131, 254, 255
  );
  db852po0 :PCollationTable = @_db852po0;




  // DB860PT0      Checksum: 33927

  _DB860PT0 :TCollationTable = (
    000, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016, 
    017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 032, 
    001, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 
    158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 048, 049, 050, 051, 052, 053, 
    054, 168, 178, 180, 184, 186, 194, 196, 198, 200, 206, 208, 210, 212, 214, 218, 
    228, 230, 232, 234, 236, 238, 246, 248, 250, 252, 254, 055, 056, 057, 058, 059, 
    060, 173, 179, 182, 185, 190, 195, 197, 199, 203, 207, 209, 211, 213, 215, 223, 
    229, 231, 233, 235, 237, 242, 247, 249, 251, 253, 255, 061, 062, 063, 064, 065, 
    181, 244, 192, 174, 177, 176, 170, 183, 191, 187, 193, 201, 219, 205, 172, 169, 
    188, 171, 189, 224, 226, 227, 239, 245, 202, 221, 240, 066, 067, 241, 068, 220, 
    175, 204, 225, 243, 217, 216, 069, 070, 071, 222, 072, 073, 074, 075, 076, 077, 
    078, 079, 080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 090, 091, 092, 093, 
    094, 095, 096, 097, 098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 
    110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 
    126, 127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 
    142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157
  );
  DB860PT0 :PCollationTable = @_DB860PT0;




  // DB850PT0      Checksum: 5850

  _DB850PT0 :TCollationTable = (
    000, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016, 
    017, 018, 019, 020, 021, 023, 025, 026, 027, 028, 029, 030, 031, 032, 033, 034, 
    001, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 048, 049, 
    141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 050, 051, 052, 053, 054, 055, 
    056, 151, 165, 167, 171, 173, 183, 185, 187, 189, 199, 201, 203, 205, 207, 211, 
    223, 225, 227, 229, 231, 233, 243, 245, 247, 249, 254, 057, 058, 059, 060, 061, 
    062, 158, 166, 169, 172, 178, 184, 186, 188, 194, 200, 202, 204, 206, 208, 217, 
    224, 226, 228, 230, 232, 238, 244, 246, 248, 251, 255, 063, 064, 065, 066, 067, 
    168, 240, 180, 159, 163, 161, 164, 170, 179, 182, 181, 198, 196, 197, 156, 157, 
    176, 068, 069, 218, 222, 221, 242, 241, 253, 216, 235, 070, 071, 072, 073, 074, 
    160, 195, 219, 239, 210, 209, 075, 076, 077, 078, 079, 080, 081, 082, 083, 084, 
    085, 086, 087, 088, 089, 153, 152, 154, 090, 091, 092, 093, 094, 095, 096, 097, 
    098, 099, 100, 101, 102, 103, 162, 155, 104, 105, 106, 107, 108, 109, 110, 111, 
    112, 113, 174, 177, 175, 114, 190, 191, 193, 115, 116, 117, 118, 119, 192, 120, 
    213, 121, 212, 214, 220, 215, 122, 123, 124, 234, 237, 236, 252, 250, 125, 126, 
    127, 128, 129, 130, 022, 024, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140
  );
  DB850PT0 :PCollationTable = @_DB850PT0;




  // db866ru0      Checksum: 43413

  _db866ru0 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 
    064, 065, 066, 067, 068, 069, 070, 071, 072, 073, 074, 075, 076, 077, 078, 079, 
    080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 090, 091, 092, 093, 094, 095, 
    096, 097, 098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 
    112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 
    128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 
    144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 
    160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 
    176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 
    192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 
    208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 
    224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  db866ru0 :PCollationTable = @_db866ru0;


{$ifdef USE_BORLAND_COLLATION_TABLES}

  // BLLT1DA0    64770

  _BLLT1DA0 :TCollationTable = (
    000, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020,
    020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020,
    020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020,
    002, 006, 008, 010, 012, 013, 014, 015, 016, 017, 020, 020, 020, 020, 020, 020,
    020, 018, 030, 032, 036, 040, 050, 052, 054, 056, 066, 068, 070, 072, 074, 078,
    089, 091, 093, 095, 098, 102, 110, 112, 114, 116, 123, 020, 020, 020, 020, 020,
    020, 019, 031, 033, 037, 041, 051, 053, 055, 057, 067, 069, 071, 073, 075, 079,
    090, 092, 094, 096, 099, 103, 111, 113, 115, 117, 124, 020, 020, 020, 020, 020,
    135, 136, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 137, 138, 139,
    140, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 141, 142, 020,
    001, 020, 020, 020, 020, 020, 020, 020, 020, 020, 021, 020, 020, 020, 020, 020,
    020, 020, 009, 011, 020, 020, 020, 020, 020, 007, 080, 020, 003, 004, 005, 020,
    024, 022, 026, 028, 127, 133, 125, 034, 044, 042, 046, 048, 060, 058, 062, 064,
    038, 076, 083, 081, 085, 087, 131, 020, 129, 106, 104, 108, 121, 118, 100, 097,
    025, 023, 027, 029, 128, 134, 126, 035, 045, 043, 047, 049, 061, 059, 063, 065,
    039, 077, 084, 082, 086, 088, 132, 020, 130, 107, 105, 109, 122, 119, 101, 120
  );
  BLLT1DA0 :PCollationTable = @_BLLT1DA0;




  // BLLT1NL0    30499

  _BLLT1NL0 :TCollationTable = (
    000, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018,
    018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018,
    018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018,
    002, 006, 008, 010, 012, 013, 014, 015, 016, 017, 018, 018, 018, 018, 018, 018,
    018, 020, 037, 039, 043, 047, 057, 059, 061, 063, 073, 075, 077, 079, 081, 085,
    100, 102, 104, 106, 109, 113, 123, 125, 127, 129, 134, 018, 018, 018, 018, 018,
    018, 019, 036, 038, 042, 046, 056, 058, 060, 062, 072, 074, 076, 078, 080, 084,
    099, 101, 103, 105, 108, 112, 122, 124, 126, 128, 133, 018, 018, 018, 018, 018,
    135, 136, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 137, 138, 139,
    140, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 141, 142, 018,
    001, 018, 018, 018, 018, 018, 018, 018, 018, 018, 021, 018, 018, 018, 018, 018,
    018, 018, 009, 011, 018, 018, 018, 018, 018, 007, 086, 018, 003, 004, 005, 018,
    025, 023, 027, 033, 031, 029, 035, 041, 051, 049, 053, 055, 067, 065, 069, 071,
    045, 083, 090, 088, 092, 096, 094, 018, 098, 117, 115, 119, 121, 131, 111, 107,
    024, 022, 026, 032, 030, 028, 034, 040, 050, 048, 052, 054, 066, 064, 068, 070,
    044, 082, 089, 087, 091, 095, 093, 018, 097, 116, 114, 118, 120, 130, 110, 132
  );
  BLLT1NL0 :PCollationTable = @_BLLT1NL0;

  BLLT1CA0 :PCollationTable = @_BLLT1NL0;

  BLLT1IT0 :PCollationTable = @_BLLT1NL0;

  BLLT1ES0 :PCollationTable = @_BLLT1NL0;

  BLLT1UK0 :PCollationTable = @_BLLT1NL0;

  BLLT1PT0 :PCollationTable = @_BLLT1NL0;




  // BLLT1FI0    29479

  _BLLT1FI0 :TCollationTable = (
    000, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018,
    018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018,
    018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018,
    002, 006, 008, 010, 012, 013, 014, 015, 016, 017, 018, 018, 018, 018, 018, 018,
    018, 020, 031, 033, 037, 041, 051, 053, 055, 057, 067, 069, 071, 073, 075, 079,
    090, 092, 094, 096, 099, 103, 111, 113, 115, 117, 124, 018, 018, 018, 018, 018,
    018, 019, 030, 032, 036, 040, 050, 052, 054, 056, 066, 068, 070, 072, 074, 078,
    089, 091, 093, 095, 098, 102, 110, 112, 114, 116, 123, 018, 018, 018, 018, 018,
    135, 136, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 137, 138, 139,
    140, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 141, 142, 018,
    001, 018, 018, 018, 018, 018, 018, 018, 018, 018, 021, 018, 018, 018, 018, 018,
    018, 018, 009, 011, 018, 018, 018, 018, 018, 007, 080, 018, 003, 004, 005, 018,
    025, 023, 027, 029, 128, 126, 130, 035, 045, 043, 047, 049, 061, 059, 063, 065,
    039, 077, 084, 082, 086, 088, 132, 018, 134, 107, 105, 109, 122, 119, 101, 097,
    024, 022, 026, 028, 127, 125, 129, 034, 044, 042, 046, 048, 060, 058, 062, 064,
    038, 076, 083, 081, 085, 087, 131, 018, 133, 106, 104, 108, 121, 118, 100, 120
  );
  BLLT1FI0 :PCollationTable = @_BLLT1FI0;

  BLLT1SV0 :PCollationTable = @_BLLT1FI0;




  // BLLT1FR0    48778

  _BLLT1FR0 :TCollationTable = (
    000, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 032,
    033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 048,
    049, 057, 071, 093, 087, 094, 092, 070, 074, 075, 090, 095, 054, 053, 062, 061,
    002, 006, 008, 010, 012, 013, 014, 015, 016, 017, 056, 055, 099, 100, 101, 059,
    084, 133, 150, 152, 156, 160, 170, 172, 174, 176, 186, 188, 190, 192, 194, 198,
    213, 215, 217, 219, 222, 226, 236, 238, 240, 242, 247, 076, 091, 077, 065, 050,
    064, 132, 149, 151, 155, 159, 169, 171, 173, 175, 185, 187, 189, 191, 193, 197,
    212, 214, 216, 218, 221, 225, 235, 237, 239, 241, 246, 078, 103, 079, 067, 107,
    248, 249, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 250, 251, 252,
    253, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129, 130, 254, 255, 131,
    001, 058, 086, 088, 085, 089, 104, 080, 066, 082, 134, 072, 102, 052, 083, 051,
    105, 096, 009, 011, 063, 106, 081, 068, 069, 007, 199, 073, 003, 004, 005, 060,
    138, 136, 140, 146, 144, 142, 148, 154, 164, 162, 166, 168, 180, 178, 182, 184,
    158, 196, 203, 201, 205, 209, 207, 098, 211, 230, 228, 232, 234, 244, 224, 220,
    137, 135, 139, 145, 143, 141, 147, 153, 163, 161, 165, 167, 179, 177, 181, 183,
    157, 195, 202, 200, 204, 208, 206, 097, 210, 229, 227, 231, 233, 243, 223, 245
  );
  BLLT1FR0 :PCollationTable = @_BLLT1FR0;

  BLLT1DE0 :PCollationTable = @_BLLT1FR0;




  // BLLT1IS0    64857

  _BLLT1IS0 :TCollationTable = (
    000, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018,
    018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018,
    018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018,
    002, 006, 008, 010, 012, 013, 014, 015, 016, 017, 018, 018, 018, 018, 018, 018,
    018, 020, 035, 037, 041, 045, 055, 057, 059, 061, 071, 073, 075, 077, 079, 083,
    094, 096, 098, 100, 103, 105, 115, 117, 119, 121, 126, 018, 018, 018, 018, 018,
    018, 019, 034, 036, 040, 044, 054, 056, 058, 060, 070, 072, 074, 076, 078, 082,
    093, 095, 097, 099, 102, 104, 114, 116, 118, 120, 125, 018, 018, 018, 018, 018,
    135, 136, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 137, 138, 139,
    140, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 018, 141, 142, 018,
    001, 018, 018, 018, 018, 018, 018, 018, 018, 018, 021, 018, 018, 018, 018, 018,
    018, 018, 009, 011, 018, 018, 018, 018, 018, 007, 084, 018, 003, 004, 005, 018,
    023, 033, 025, 031, 029, 027, 130, 039, 047, 053, 049, 051, 063, 069, 065, 067,
    043, 081, 086, 092, 088, 090, 132, 018, 134, 107, 113, 109, 111, 124, 128, 101,
    022, 032, 024, 030, 028, 026, 129, 038, 046, 052, 048, 050, 062, 068, 064, 066,
    042, 080, 085, 091, 087, 089, 131, 018, 133, 106, 112, 108, 110, 123, 127, 122
  );
  BLLT1IS0 :PCollationTable = @_BLLT1IS0;




  // BLLT1NO0    17250

  _BLLT1NO0 :TCollationTable = (
    000, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020,
    020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020,
    020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020,
    002, 006, 008, 010, 012, 013, 014, 015, 016, 017, 020, 020, 020, 020, 020, 020,
    020, 019, 033, 035, 039, 043, 053, 055, 057, 059, 069, 071, 073, 075, 077, 081,
    094, 096, 098, 100, 103, 107, 117, 119, 121, 123, 128, 020, 020, 020, 020, 020,
    020, 018, 032, 034, 038, 042, 052, 054, 056, 058, 068, 070, 072, 074, 076, 080,
    093, 095, 097, 099, 102, 106, 116, 118, 120, 122, 127, 020, 020, 020, 020, 020,
    135, 136, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 137, 138, 139,
    140, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 020, 141, 142, 020,
    001, 020, 020, 020, 020, 020, 020, 020, 020, 020, 021, 020, 020, 020, 020, 020,
    020, 020, 009, 011, 020, 020, 020, 020, 020, 007, 082, 020, 003, 004, 005, 020,
    025, 023, 027, 031, 029, 134, 130, 037, 047, 045, 049, 051, 063, 061, 065, 067,
    041, 079, 086, 084, 088, 092, 090, 020, 132, 111, 109, 113, 115, 125, 105, 101,
    024, 022, 026, 030, 028, 133, 129, 036, 046, 044, 048, 050, 062, 060, 064, 066,
    040, 078, 085, 083, 087, 091, 089, 020, 131, 110, 108, 112, 114, 124, 104, 126
  );
  BLLT1NO0 :PCollationTable = @_BLLT1NO0;

{$endif}


  // DB850US0      Checksum: 43413

  _DB850US0 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 
    064, 065, 066, 067, 068, 069, 070, 071, 072, 073, 074, 075, 076, 077, 078, 079, 
    080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 090, 091, 092, 093, 094, 095, 
    096, 097, 098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 
    112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 
    128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 
    144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 
    160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 
    176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 
    192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 
    208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 
    224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  DB850US0 :PCollationTable = @_DB850US0;




{$ifdef USE_PARADOX_COLLATIONS}

  // intl850    43039

  _intl850 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 073, 083, 086, 090, 097, 103, 105, 107, 113, 119, 121, 123, 125, 128, 137,
    146, 148, 150, 152, 155, 162, 169, 171, 173, 177, 180, 181, 182, 183, 184, 185,
    186, 065, 082, 084, 088, 092, 102, 104, 106, 108, 118, 120, 122, 124, 126, 130,
    145, 147, 149, 151, 154, 158, 168, 170, 172, 174, 179, 187, 188, 189, 190, 191,
    035, 035, 035, 194, 035, 035, 035, 035, 035, 035, 152, 035, 137, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 151, 035, 130, 035, 035, 177,
    224, 200, 204, 192, 206, 205, 207, 215, 219, 203, 066, 201, 197, 211, 196, 209,
    218, 212, 223, 222, 210, 208, 214, 220, 217, 221, 131, 202, 199, 198, 213, 195,
    076, 075, 077, 079, 081, 078, 074, 087, 100, 099, 101, 098, 116, 115, 117, 114,
    091, 129, 139, 138, 140, 141, 144, 193, 142, 164, 163, 165, 167, 178, 157, 153,
    069, 068, 070, 072, 080, 071, 067, 085, 095, 094, 096, 093, 111, 110, 112, 109,
    089, 127, 133, 132, 134, 135, 143, 216, 136, 160, 159, 161, 166, 176, 156, 175
  );
  intl850 :PCollationTable = @_intl850;


  // SPANISH    20109

  _SPANISH :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 048,
    049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 064,
    066, 074, 080, 083, 086, 092, 095, 097, 099, 105, 107, 109, 111, 113, 115, 124,
    127, 129, 131, 133, 136, 142, 145, 147, 149, 152, 154, 155, 156, 157, 158, 159,
    160, 067, 079, 081, 085, 087, 094, 096, 098, 100, 106, 108, 110, 112, 114, 118,
    126, 128, 130, 132, 135, 137, 144, 146, 148, 150, 153, 161, 162, 163, 164, 165,
    036, 036, 036, 169, 036, 036, 036, 036, 036, 036, 133, 036, 124, 036, 036, 036,
    036, 036, 036, 036, 036, 036, 036, 036, 036, 036, 132, 036, 118, 036, 036, 152,
    181, 034, 166, 167, 036, 168, 036, 036, 036, 036, 068, 173, 170, 036, 036, 036,
    178, 176, 180, 036, 036, 175, 036, 179, 036, 036, 119, 174, 172, 171, 036, 065,
    074, 074, 074, 074, 075, 076, 078, 084, 092, 093, 092, 092, 105, 105, 105, 105,
    086, 117, 124, 124, 124, 124, 125, 036, 124, 142, 142, 142, 143, 152, 036, 134,
    070, 069, 071, 067, 072, 073, 077, 082, 089, 088, 090, 091, 102, 101, 103, 104,
    085, 116, 121, 120, 122, 118, 123, 177, 118, 139, 138, 140, 141, 150, 036, 151
  );
  SPANISH :PCollationTable = @_SPANISH;






  // iceland    23936

  _iceland :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 070, 076, 079, 082, 089, 093, 095, 097, 099, 103, 105, 107, 109, 111, 114,
    118, 120, 122, 124, 126, 130, 135, 137, 139, 141, 145, 154, 155, 156, 157, 158,
    159, 065, 075, 077, 081, 085, 092, 094, 096, 098, 102, 104, 106, 108, 110, 112,
    117, 119, 121, 123, 125, 127, 134, 136, 138, 140, 144, 160, 161, 162, 163, 035,
    035, 035, 035, 165, 035, 035, 035, 035, 035, 035, 124, 035, 114, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 123, 035, 112, 035, 035, 141,
    180, 170, 035, 164, 035, 035, 035, 035, 035, 035, 035, 171, 167, 035, 035, 035,
    177, 175, 179, 035, 035, 174, 035, 178, 035, 035, 035, 172, 169, 168, 035, 166,
    070, 074, 070, 070, 072, 071, 149, 080, 089, 091, 089, 089, 099, 101, 099, 099,
    084, 111, 114, 116, 114, 114, 152, 035, 153, 130, 133, 130, 131, 143, 147, 173,
    066, 073, 067, 065, 069, 068, 148, 078, 086, 090, 087, 088, 098, 100, 098, 098,
    083, 110, 112, 115, 113, 112, 150, 176, 151, 127, 132, 128, 129, 142, 146, 140
  );
  iceland :PCollationTable = @_iceland;






  // ANSIINTL    58462

  _ANSIINTL :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 070, 078, 081, 085, 092, 098, 100, 102, 108, 114, 116, 118, 120, 123, 131,
    140, 142, 144, 146, 149, 156, 163, 165, 167, 171, 174, 177, 178, 179, 180, 181,
    182, 065, 077, 079, 083, 087, 097, 099, 101, 103, 113, 115, 117, 119, 121, 125,
    139, 141, 143, 145, 148, 152, 162, 164, 166, 168, 173, 183, 184, 185, 186, 187,
    248, 249, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 250, 251, 252,
    253, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 254, 255, 213,
    214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229,
    230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245,
    072, 071, 073, 074, 076, 176, 189, 082, 095, 094, 096, 093, 111, 110, 112, 109,
    086, 124, 133, 132, 134, 135, 138, 246, 136, 158, 157, 159, 161, 172, 151, 147,
    067, 066, 068, 069, 075, 175, 188, 080, 090, 089, 091, 088, 106, 105, 107, 104,
    084, 122, 127, 126, 128, 129, 137, 247, 130, 154, 153, 155, 160, 170, 150, 169
  );
  ANSIINTL :PCollationTable = @_ANSIINTL;






  // ANSII850    29000

  _ANSII850 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 073, 083, 086, 090, 097, 103, 105, 107, 113, 119, 121, 123, 125, 128, 137,
    146, 148, 150, 152, 155, 162, 169, 171, 173, 177, 180, 181, 182, 183, 184, 185,
    186, 065, 082, 084, 088, 092, 102, 104, 106, 108, 118, 120, 122, 124, 126, 130,
    145, 147, 149, 151, 154, 158, 168, 170, 172, 174, 179, 187, 188, 189, 190, 191,
    248, 249, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 250, 251, 252,
    253, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 254, 255, 215,
    216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 066, 226, 227, 228, 229, 230,
    231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 131, 241, 242, 243, 244, 245,
    076, 075, 077, 079, 081, 078, 074, 087, 100, 099, 101, 098, 116, 115, 117, 114,
    091, 129, 139, 138, 140, 141, 144, 246, 142, 164, 163, 165, 167, 178, 157, 153,
    069, 068, 070, 072, 080, 071, 067, 085, 095, 094, 096, 093, 111, 110, 112, 109,
    089, 127, 133, 132, 134, 135, 143, 247, 136, 160, 159, 161, 166, 176, 156, 175
  );
  ANSII850 :PCollationTable = @_ANSII850;






  // ANSISPAN    33308

  _ANSISPAN :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 048,
    049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 064,
    066, 075, 085, 088, 092, 099, 105, 107, 109, 115, 121, 123, 125, 127, 129, 140,
    148, 150, 152, 154, 158, 165, 171, 173, 175, 179, 182, 183, 184, 185, 186, 187,
    188, 067, 084, 086, 090, 094, 104, 106, 108, 110, 120, 122, 124, 126, 128, 132,
    147, 149, 151, 153, 156, 160, 170, 172, 174, 176, 181, 189, 190, 191, 192, 193,
    248, 249, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 250, 251, 252,
    253, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 254, 255, 217,
    218, 034, 219, 220, 221, 222, 223, 224, 225, 226, 069, 227, 228, 229, 230, 231,
    232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 133, 242, 243, 244, 245, 065,
    077, 076, 078, 079, 080, 081, 083, 089, 102, 101, 103, 100, 118, 117, 119, 116,
    093, 131, 142, 141, 143, 145, 144, 246, 146, 167, 166, 168, 169, 180, 159, 155,
    070, 068, 071, 072, 073, 074, 082, 087, 096, 095, 097, 098, 112, 111, 113, 114,
    091, 130, 135, 134, 136, 138, 137, 247, 139, 162, 161, 163, 164, 177, 157, 178
  );
  ANSISPAN :PCollationTable = @_ANSISPAN;






  // ANSISWFN    44782

  _ANSISWFN :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 070, 076, 079, 083, 090, 096, 098, 100, 106, 112, 114, 116, 118, 121, 128,
    134, 136, 138, 140, 142, 149, 154, 156, 158, 162, 165, 178, 179, 180, 181, 182,
    183, 065, 075, 077, 081, 085, 095, 097, 099, 101, 111, 113, 115, 117, 119, 123,
    133, 135, 137, 139, 141, 145, 153, 155, 157, 159, 164, 184, 185, 186, 187, 188,
    248, 249, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 250, 251, 252,
    253, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 254, 255, 212,
    213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228,
    229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244,
    072, 071, 073, 074, 173, 171, 167, 080, 093, 092, 094, 091, 109, 108, 110, 107,
    084, 122, 130, 129, 131, 132, 175, 245, 169, 151, 150, 152, 177, 163, 144, 246,
    067, 066, 068, 069, 172, 170, 166, 078, 088, 087, 089, 086, 104, 103, 105, 102,
    082, 120, 125, 124, 126, 127, 174, 247, 168, 147, 146, 148, 176, 160, 143, 161
  );
  ANSISWFN :PCollationTable = @_ANSISWFN;






  // ANSINOR4    55290

  _ANSINOR4 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 065, 075, 077, 081, 085, 095, 097, 099, 101, 111, 113, 115, 117, 119, 123,
    133, 135, 137, 139, 141, 145, 153, 155, 157, 159, 166, 178, 179, 180, 181, 182,
    183, 070, 076, 079, 083, 090, 096, 098, 100, 106, 112, 114, 116, 118, 121, 128,
    134, 136, 138, 140, 142, 149, 154, 156, 158, 162, 167, 184, 185, 186, 187, 188,
    248, 249, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 250, 251, 252,
    253, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 254, 255, 212,
    213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228,
    229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244,
    067, 066, 068, 069, 169, 176, 168, 078, 088, 087, 089, 086, 104, 103, 105, 102,
    082, 120, 125, 124, 126, 127, 173, 245, 172, 147, 146, 148, 161, 160, 143, 246,
    072, 071, 073, 074, 171, 177, 170, 080, 093, 092, 094, 091, 109, 108, 110, 107,
    084, 122, 130, 129, 131, 132, 175, 247, 174, 151, 150, 152, 165, 164, 144, 163
  );
  ANSINOR4 :PCollationTable = @_ANSINOR4;

{$endif}




  // china    54324

  _china :TCollationTable = (
    000, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142,
    143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158,
    159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174,
    175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190,
    191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206,
    207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222,
    223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238,
    239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254,
    255, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 065, 066, 067, 068, 069, 070, 071, 072, 073, 074, 075, 076, 077, 078, 079,
    080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 090, 091, 092, 093, 094, 095,
    096, 097, 098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
    112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127
  );

  DB936CN0 :PCollationTable = @_china;

  DB949KO0 :PCollationTable = @_china;

  DB950TW0 :PCollationTable = @_china;




  // thai      Checksum: 9864

  _thai :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 
    064, 065, 066, 067, 068, 069, 070, 071, 072, 073, 074, 075, 076, 077, 078, 079, 
    080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 090, 091, 092, 093, 094, 095, 
    096, 097, 098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 
    112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 
    211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 
    227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 
    128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 
    144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 
    160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 191, 
    192, 183, 193, 184, 185, 186, 187, 188, 189, 190, 175, 243, 244, 245, 246, 194, 
    247, 248, 249, 250, 251, 195, 196, 176, 177, 178, 179, 180, 181, 182, 197, 198, 
    199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 252, 253, 254, 255
  );

{$ifdef USE_PARADOX_COLLATIONS}
  china :PCollationTable = @_china;

  korea :PCollationTable = @_china;

  taiwan :PCollationTable = @_china;

  thai :PCollationTable = @_thai;
{$endif}

  db874th0 :PCollationTable = @_thai;




  // DB932JP1      Checksum: 8911

  _DB932JP1 :TCollationTable = (
    000, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 
    110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 
    065, 075, 118, 135, 133, 134, 136, 117, 119, 120, 137, 127, 069, 128, 070, 114, 
    139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 072, 073, 130, 129, 131, 074, 
    138, 149, 151, 153, 155, 157, 159, 161, 163, 165, 167, 169, 171, 173, 175, 177, 
    179, 181, 183, 185, 187, 189, 191, 193, 195, 197, 199, 121, 132, 122, 078, 111, 
    112, 150, 152, 154, 156, 158, 160, 162, 164, 166, 168, 170, 172, 174, 176, 178, 
    180, 182, 184, 186, 188, 190, 192, 194, 196, 198, 200, 123, 116, 124, 115, 110, 
    001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016, 
    017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 032, 
    066, 068, 125, 126, 067, 071, 254, 201, 203, 205, 207, 209, 242, 244, 246, 223, 
    113, 202, 204, 206, 208, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 
    221, 222, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 
    238, 239, 240, 241, 243, 245, 247, 248, 249, 250, 251, 252, 253, 255, 076, 077, 
    033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 048, 
    049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 064
  );
  DB932JP1 :PCollationTable = @_DB932JP1;




  // DBWINWE0    5562

  _DBWINWE0 :TCollationTable = (
    000, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016,
    017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 032,
    001, 041, 055, 077, 071, 078, 076, 054, 058, 059, 074, 079, 038, 037, 046, 045,
    111, 115, 117, 119, 121, 122, 123, 124, 125, 126, 040, 039, 083, 084, 085, 043,
    068, 128, 145, 147, 151, 155, 165, 167, 169, 171, 181, 183, 185, 187, 189, 193,
    210, 212, 214, 216, 221, 225, 235, 237, 239, 241, 247, 060, 075, 061, 049, 034,
    048, 127, 144, 146, 150, 154, 164, 166, 168, 170, 180, 182, 184, 186, 188, 192,
    209, 211, 213, 215, 220, 224, 234, 236, 238, 240, 246, 062, 087, 063, 051, 091,
    248, 249, 092, 093, 094, 095, 096, 097, 098, 099, 218, 100, 208, 250, 251, 252,
    253, 101, 102, 103, 104, 105, 106, 107, 108, 109, 217, 110, 207, 254, 255, 245,
    033, 042, 070, 072, 069, 073, 088, 064, 050, 066, 129, 056, 086, 036, 067, 035,
    089, 080, 118, 120, 047, 090, 065, 052, 053, 116, 194, 057, 112, 113, 114, 044,
    133, 131, 135, 141, 139, 137, 143, 149, 159, 157, 161, 163, 175, 173, 177, 179,
    153, 191, 198, 196, 200, 204, 202, 082, 206, 229, 227, 231, 233, 243, 223, 219,
    132, 130, 134, 140, 138, 136, 142, 148, 158, 156, 160, 162, 174, 172, 176, 178,
    152, 190, 197, 195, 199, 203, 201, 081, 205, 228, 226, 230, 232, 242, 222, 244
  );
  DBWINWE0 :PCollationTable = @_DBWINWE0;

  DBWINES0 :PCollationTable = @_DBWINWE0;


{$ifdef USE_ACCESS_COLLATIONS}

  // ACCGEN    19621

  _ACCGEN :TCollationTable = (
    000, 001, 001, 001, 001, 001, 001, 001, 001, 002, 001, 001, 001, 001, 001, 001,
    001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001,
    003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016, 017, 018,
    072, 073, 074, 075, 076, 077, 078, 079, 080, 081, 019, 020, 021, 022, 023, 024,
    025, 082, 090, 091, 093, 095, 100, 101, 102, 103, 108, 109, 110, 111, 112, 114,
    121, 122, 123, 124, 127, 128, 133, 134, 135, 136, 139, 026, 027, 028, 029, 030,
    031, 082, 090, 091, 093, 095, 100, 101, 102, 103, 108, 109, 110, 111, 112, 114,
    121, 122, 123, 124, 127, 128, 133, 134, 135, 136, 139, 032, 033, 034, 035, 001,
    142, 143, 010, 036, 005, 037, 038, 039, 040, 041, 125, 010, 120, 144, 145, 146,
    147, 010, 010, 005, 005, 042, 016, 016, 043, 044, 125, 010, 120, 148, 149, 138,
    003, 045, 046, 047, 048, 049, 050, 051, 052, 053, 054, 005, 055, 016, 056, 057,
    058, 059, 074, 075, 060, 061, 062, 063, 064, 073, 065, 005, 066, 067, 068, 069,
    083, 084, 085, 087, 086, 088, 089, 092, 096, 097, 098, 099, 104, 105, 106, 107,
    094, 113, 115, 116, 117, 119, 118, 070, 141, 129, 130, 131, 132, 137, 140, 126,
    083, 084, 085, 087, 086, 088, 089, 092, 096, 097, 098, 099, 104, 105, 106, 107,
    094, 113, 115, 116, 117, 119, 118, 071, 141, 129, 130, 131, 132, 137, 140, 138
  );
  ACCGEN :PCollationTable = @_ACCGEN;




  // ACCNRDAN    24816

  _ACCNRDAN :TCollationTable = (
    000, 001, 001, 001, 001, 001, 001, 001, 001, 002, 001, 001, 001, 001, 001, 001,
    001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001,
    003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016, 017, 018,
    072, 073, 074, 075, 076, 077, 078, 079, 080, 081, 019, 020, 021, 022, 023, 024,
    025, 140, 086, 087, 089, 091, 096, 097, 098, 099, 104, 105, 106, 107, 108, 110,
    116, 117, 118, 119, 122, 124, 128, 129, 130, 131, 135, 026, 027, 028, 029, 030,
    031, 140, 086, 087, 089, 091, 096, 097, 098, 099, 104, 105, 106, 107, 108, 110,
    116, 117, 118, 119, 122, 124, 128, 129, 130, 131, 135, 032, 033, 034, 035, 001,
    142, 143, 010, 036, 005, 037, 038, 039, 040, 041, 120, 010, 115, 144, 145, 146,
    147, 010, 010, 005, 005, 042, 016, 016, 043, 044, 120, 010, 115, 148, 149, 133,
    003, 045, 046, 047, 048, 049, 050, 051, 052, 053, 054, 005, 055, 016, 056, 057,
    058, 059, 074, 075, 060, 061, 062, 063, 064, 073, 065, 005, 066, 067, 068, 069,
    083, 082, 084, 085, 137, 141, 136, 088, 093, 092, 094, 095, 101, 100, 102, 103,
    090, 109, 112, 111, 113, 114, 139, 070, 138, 126, 125, 127, 134, 132, 123, 121,
    083, 082, 084, 085, 137, 141, 136, 088, 093, 092, 094, 095, 101, 100, 102, 103,
    090, 109, 112, 111, 113, 114, 139, 071, 138, 126, 125, 127, 134, 132, 123, 133
  );
  ACCNRDAN :PCollationTable = @_ACCNRDAN;




  // ACCSWFIN    39386

  _ACCSWFIN :TCollationTable = (
    000, 001, 001, 001, 001, 001, 001, 001, 001, 002, 001, 001, 001, 001, 001, 001,
    001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001,
    003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016, 017, 018,
    072, 073, 074, 075, 076, 077, 078, 079, 080, 081, 019, 020, 021, 022, 023, 024,
    025, 082, 088, 089, 091, 093, 098, 099, 100, 101, 106, 107, 108, 109, 110, 112,
    118, 119, 120, 121, 124, 126, 130, 131, 132, 133, 137, 026, 027, 028, 029, 030,
    031, 082, 088, 089, 091, 093, 098, 099, 100, 101, 106, 107, 108, 109, 110, 112,
    118, 119, 120, 121, 124, 126, 130, 131, 132, 133, 137, 032, 033, 034, 035, 001,
    142, 143, 010, 036, 005, 037, 038, 039, 040, 041, 122, 010, 117, 144, 145, 146,
    147, 010, 010, 005, 005, 042, 016, 016, 043, 044, 122, 010, 117, 148, 149, 135,
    003, 045, 046, 047, 048, 049, 050, 051, 052, 053, 054, 005, 055, 016, 056, 057,
    058, 059, 074, 075, 060, 061, 062, 063, 064, 073, 065, 005, 066, 067, 068, 069,
    084, 083, 085, 086, 139, 138, 087, 090, 095, 094, 096, 097, 103, 102, 104, 105,
    092, 111, 114, 113, 115, 116, 140, 070, 141, 128, 127, 129, 136, 134, 125, 123,
    084, 083, 085, 086, 139, 138, 087, 090, 095, 094, 096, 097, 103, 102, 104, 105,
    092, 111, 114, 113, 115, 116, 140, 071, 141, 128, 127, 129, 136, 134, 125, 135
  );
  ACCSWFIN :PCollationTable = @_ACCSWFIN;

{$endif}


  // FOXDE437      Checksum: 21075

  _FOXDE437 :TCollationTable = (
    000, 125, 125, 125, 125, 125, 125, 125, 125, 127, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 
    160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 143, 144, 145, 146, 147, 148, 
    149, 171, 182, 184, 188, 190, 197, 199, 201, 203, 209, 211, 213, 215, 217, 221, 
    226, 228, 230, 232, 237, 239, 246, 248, 250, 252, 255, 150, 151, 152, 153, 154, 
    155, 171, 182, 184, 188, 190, 197, 199, 201, 203, 209, 211, 213, 215, 217, 221, 
    226, 228, 230, 232, 237, 239, 246, 248, 250, 252, 255, 156, 157, 158, 159, 125, 
    186, 244, 192, 174, 180, 173, 176, 186, 194, 195, 193, 207, 206, 205, 180, 176, 
    192, 180, 180, 224, 235, 223, 242, 241, 253, 235, 244, 125, 125, 125, 125, 125, 
    172, 204, 222, 240, 219, 219, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 235, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125
  );
  FOXDE437 :PCollationTable = @_FOXDE437;




  // FOXNO437      Checksum: 473

  _FOXNO437 :TCollationTable = (
    000, 125, 125, 125, 125, 125, 125, 125, 125, 127, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 
    160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 143, 144, 145, 146, 147, 148, 
    149, 171, 176, 178, 182, 184, 191, 193, 195, 197, 203, 205, 207, 209, 211, 215, 
    220, 222, 224, 226, 229, 231, 236, 238, 240, 242, 247, 150, 151, 152, 153, 154, 
    155, 171, 176, 178, 182, 184, 191, 193, 195, 197, 203, 205, 207, 209, 211, 215, 
    220, 222, 224, 226, 229, 231, 236, 238, 240, 242, 247, 156, 157, 158, 159, 125, 
    180, 245, 187, 174, 251, 172, 255, 180, 188, 189, 185, 201, 200, 198, 251, 255, 
    187, 249, 249, 218, 253, 216, 234, 232, 245, 253, 245, 125, 125, 125, 125, 125, 
    173, 199, 217, 233, 213, 213, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 227, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 
    125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125
  );
  FOXNO437 :PCollationTable = @_FOXNO437 ;




  // FOXNO850      Checksum: 62

  _FOXNO850 :TCollationTable = (
    000, 096, 096, 096, 096, 096, 096, 096, 096, 098, 096, 096, 096, 096, 096, 096, 
    096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 
    098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 
    132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 114, 115, 116, 117, 118, 119, 
    120, 143, 153, 155, 159, 163, 173, 175, 177, 179, 189, 191, 193, 195, 197, 201, 
    211, 213, 215, 217, 220, 222, 230, 232, 234, 236, 243, 121, 122, 123, 124, 125, 
    126, 143, 153, 155, 159, 163, 173, 175, 177, 179, 189, 191, 193, 195, 197, 201, 
    211, 213, 215, 217, 220, 222, 230, 232, 234, 236, 243, 127, 128, 129, 130, 096, 
    157, 241, 167, 149, 249, 145, 255, 157, 169, 171, 165, 187, 185, 181, 249, 255, 
    167, 247, 247, 207, 253, 203, 228, 224, 241, 253, 241, 251, 096, 251, 096, 096, 
    147, 183, 205, 226, 199, 199, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 
    096, 096, 096, 096, 096, 147, 149, 145, 131, 096, 096, 096, 096, 096, 096, 096, 
    096, 096, 096, 096, 096, 096, 151, 151, 096, 096, 096, 096, 096, 096, 096, 096, 
    161, 161, 169, 171, 165, 096, 183, 185, 187, 096, 096, 096, 096, 096, 181, 096, 
    205, 218, 207, 203, 209, 209, 096, 245, 245, 226, 228, 224, 238, 238, 096, 096, 
    096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096, 096
  );
  FOXNO850 :PCollationTable = @_FOXNO850;




  // FOXDEWIN    44890

  _FOXDEWIN :TCollationTable = (
    000, 001, 001, 001, 001, 001, 001, 001, 001, 002, 001, 001, 001, 001, 001, 001,
    001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001,
    002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016, 017,
    072, 073, 074, 075, 076, 077, 078, 079, 080, 081, 019, 020, 021, 022, 023, 024,
    025, 082, 089, 090, 092, 094, 099, 100, 101, 102, 107, 108, 109, 110, 111, 113,
    119, 120, 121, 122, 125, 127, 132, 133, 134, 135, 138, 026, 027, 028, 029, 030,
    031, 082, 089, 090, 092, 094, 099, 100, 101, 102, 107, 108, 109, 110, 111, 113,
    119, 120, 121, 122, 125, 127, 132, 133, 134, 135, 138, 032, 033, 034, 035, 001,
    140, 141, 009, 036, 004, 037, 038, 039, 040, 041, 123, 009, 118, 142, 143, 144,
    145, 009, 009, 004, 004, 042, 015, 015, 043, 044, 123, 009, 118, 146, 147, 137,
    018, 045, 046, 047, 048, 049, 050, 051, 052, 053, 054, 004, 055, 015, 056, 057,
    058, 059, 074, 075, 060, 061, 062, 063, 064, 073, 065, 004, 066, 067, 068, 069,
    084, 083, 085, 086, 088, 087, 088, 091, 096, 095, 097, 098, 104, 103, 105, 106,
    093, 112, 115, 114, 116, 117, 118, 070, 139, 129, 128, 130, 131, 136, 126, 124,
    084, 083, 085, 086, 088, 087, 088, 091, 096, 095, 097, 098, 104, 103, 105, 106,
    093, 112, 115, 114, 116, 117, 118, 071, 139, 129, 128, 130, 131, 136, 126, 137
  );
  FOXDEWIN :PCollationTable = @_FOXDEWIN;




  // FOXNOWIN      Checksum: 58863

  _FOXNOWIN :TCollationTable = (
    000, 031, 031, 031, 031, 031, 031, 031, 031, 033, 031, 031, 031, 031, 031, 031, 
    031, 031, 031, 031, 031, 031, 031, 031, 031, 031, 031, 031, 031, 031, 031, 031, 
    033, 035, 041, 042, 043, 044, 045, 051, 052, 053, 054, 055, 056, 060, 061, 062, 
    116, 118, 120, 122, 123, 124, 125, 126, 127, 128, 063, 064, 065, 066, 067, 068, 
    069, 130, 140, 142, 146, 150, 160, 162, 164, 166, 176, 178, 180, 182, 184, 188, 
    200, 202, 204, 206, 211, 213, 223, 225, 227, 229, 235, 070, 071, 072, 073, 074, 
    075, 130, 140, 142, 146, 150, 160, 162, 164, 166, 176, 178, 180, 182, 184, 188, 
    200, 202, 204, 206, 211, 213, 223, 225, 227, 229, 235, 076, 077, 078, 079, 031, 
    248, 249, 051, 080, 041, 081, 082, 083, 084, 085, 208, 051, 198, 250, 251, 252, 
    253, 051, 051, 041, 041, 086, 060, 060, 087, 088, 208, 051, 198, 254, 255, 233, 
    034, 089, 090, 091, 092, 093, 094, 095, 096, 097, 098, 041, 099, 060, 100, 101, 
    102, 103, 120, 122, 104, 105, 106, 107, 108, 118, 109, 041, 110, 111, 112, 113, 
    134, 132, 136, 138, 241, 247, 239, 144, 154, 152, 156, 158, 170, 168, 172, 174, 
    148, 186, 192, 190, 194, 196, 245, 114, 243, 217, 215, 219, 221, 231, 237, 209, 
    134, 132, 136, 138, 241, 247, 239, 144, 154, 152, 156, 158, 170, 168, 172, 174, 
    148, 186, 192, 190, 194, 196, 245, 115, 243, 217, 215, 219, 221, 231, 237, 233
  );
  FOXNOWIN :PCollationTable = @_FOXNOWIN;




{$ifdef USE_PARADOX_COLLATIONS}

  // czech    30844

  _czech :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 074, 076, 080, 082, 088, 090, 092, 094, 100, 102, 104, 106, 108, 110,
    118, 120, 122, 124, 129, 131, 137, 139, 141, 143, 147, 148, 149, 150, 151, 152,
    153, 065, 073, 075, 079, 081, 087, 089, 091, 093, 099, 101, 103, 105, 107, 109,
    117, 119, 121, 123, 128, 130, 136, 138, 140, 142, 146, 154, 155, 156, 157, 035,
    035, 035, 035, 087, 035, 035, 035, 035, 035, 035, 127, 035, 110, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 126, 035, 109, 035, 035, 143,
    170, 035, 035, 035, 162, 035, 035, 165, 169, 035, 035, 160, 159, 164, 035, 035,
    168, 035, 035, 035, 163, 035, 035, 035, 167, 035, 035, 161, 035, 035, 035, 035,
    066, 068, 070, 066, 072, 066, 066, 078, 082, 084, 082, 086, 094, 096, 098, 094,
    080, 108, 110, 112, 114, 110, 116, 158, 110, 131, 133, 131, 135, 145, 035, 125,
    065, 067, 069, 065, 071, 065, 065, 077, 081, 083, 081, 085, 093, 095, 097, 093,
    079, 107, 109, 111, 113, 109, 115, 166, 109, 130, 132, 130, 134, 144, 035, 142
  );
  czech :PCollationTable = @_czech;




  // czechw    30844

  czechw :PCollationTable = @_czech;





  // il2czw    13611

  _il2czw :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 074, 076, 080, 082, 088, 090, 092, 094, 100, 102, 104, 106, 108, 110,
    118, 120, 122, 124, 129, 131, 137, 139, 141, 143, 147, 148, 149, 150, 151, 152,
    153, 065, 073, 075, 079, 081, 087, 089, 091, 093, 099, 101, 103, 105, 107, 109,
    117, 119, 121, 123, 128, 130, 136, 138, 140, 142, 146, 154, 155, 156, 157, 035,
    035, 035, 035, 087, 035, 035, 035, 035, 035, 035, 127, 035, 110, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 126, 035, 109, 035, 035, 143,
    158, 035, 035, 035, 159, 035, 035, 160, 161, 035, 035, 035, 035, 162, 035, 035,
    163, 035, 035, 035, 164, 035, 035, 035, 165, 035, 035, 035, 035, 035, 035, 035,
    066, 068, 070, 066, 072, 066, 066, 078, 082, 084, 082, 086, 094, 096, 098, 094,
    080, 108, 110, 112, 114, 110, 116, 166, 110, 131, 133, 131, 135, 145, 035, 125,
    065, 067, 069, 065, 071, 065, 065, 077, 081, 083, 081, 085, 093, 095, 097, 093,
    079, 107, 109, 111, 113, 109, 115, 167, 109, 130, 132, 130, 134, 144, 035, 142
  );
  il2czw :PCollationTable = @_il2czw;





  // polish    59020

  _polish :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 074, 076, 080, 082, 088, 090, 092, 094, 100, 102, 104, 106, 108, 110,
    118, 120, 122, 124, 129, 131, 137, 139, 141, 143, 147, 148, 149, 150, 151, 152,
    153, 065, 073, 075, 079, 081, 087, 089, 091, 093, 099, 101, 103, 105, 107, 109,
    117, 119, 121, 123, 128, 130, 136, 138, 140, 142, 146, 154, 155, 156, 157, 035,
    035, 035, 035, 087, 035, 035, 035, 035, 035, 035, 126, 035, 110, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 125, 035, 109, 035, 035, 143,
    170, 035, 035, 035, 162, 035, 035, 165, 169, 035, 035, 160, 159, 164, 035, 035,
    168, 035, 035, 035, 163, 035, 035, 035, 167, 035, 035, 161, 035, 035, 035, 035,
    066, 068, 070, 066, 072, 066, 066, 078, 082, 084, 082, 086, 094, 096, 098, 094,
    080, 108, 110, 112, 114, 110, 116, 158, 110, 131, 133, 131, 135, 145, 035, 127,
    065, 067, 069, 065, 071, 065, 065, 077, 081, 083, 081, 085, 093, 095, 097, 093,
    079, 107, 109, 111, 113, 109, 115, 166, 109, 130, 132, 130, 134, 144, 035, 142
  );
  polish :PCollationTable = @_polish;






  // cyrr    20081

  _cyrr :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 068, 070, 072, 074, 076, 078, 080, 082, 084, 086, 088, 090, 092, 094,
    096, 098, 100, 102, 104, 106, 108, 110, 112, 114, 116, 117, 118, 119, 120, 121,
    122, 065, 067, 069, 071, 073, 075, 077, 079, 081, 083, 085, 087, 089, 091, 093,
    095, 097, 099, 101, 103, 105, 107, 109, 111, 113, 115, 123, 124, 125, 126, 127,
    035, 035, 035, 075, 035, 035, 035, 035, 035, 035, 102, 035, 094, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 101, 035, 093, 035, 035, 114,
    131, 035, 035, 035, 130, 035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 035,
    128, 035, 035, 035, 035, 035, 035, 129, 035, 035, 035, 035, 035, 035, 035, 035,
    066, 066, 066, 066, 066, 066, 066, 070, 074, 074, 074, 074, 082, 082, 082, 082,
    072, 092, 094, 094, 094, 094, 094, 035, 094, 106, 106, 106, 106, 114, 035, 035,
    065, 065, 065, 065, 065, 065, 065, 069, 073, 073, 073, 073, 081, 081, 081, 081,
    071, 091, 093, 093, 093, 093, 093, 035, 093, 105, 105, 105, 105, 113, 035, 113
  );
  cyrr :PCollationTable = @_cyrr;






  // hun852dc    62898

  _hun852dc :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 070, 072, 074, 076, 080, 082, 084, 086, 090, 092, 094, 096, 098, 100,
    107, 109, 111, 113, 115, 117, 123, 125, 127, 129, 131, 132, 133, 134, 135, 136,
    137, 065, 069, 071, 073, 075, 079, 081, 083, 085, 089, 091, 093, 095, 097, 099,
    106, 108, 110, 112, 114, 116, 122, 124, 126, 128, 130, 138, 139, 140, 141, 035,
    035, 035, 035, 079, 035, 035, 035, 035, 035, 035, 159, 035, 100, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 160, 035, 099, 035, 035, 129,
    170, 035, 035, 035, 154, 035, 035, 165, 169, 035, 035, 151, 150, 164, 035, 035,
    168, 035, 035, 035, 163, 035, 035, 035, 167, 035, 035, 152, 035, 035, 035, 035,
    066, 068, 153, 066, 148, 066, 066, 142, 076, 078, 076, 155, 086, 088, 156, 086,
    074, 098, 100, 102, 158, 100, 105, 149, 100, 117, 119, 117, 121, 162, 035, 157,
    065, 067, 143, 065, 144, 065, 065, 145, 075, 077, 075, 146, 085, 087, 147, 085,
    073, 097, 099, 101, 104, 099, 103, 166, 099, 116, 118, 116, 120, 161, 035, 128
  );
  hun852dc :PCollationTable = @_hun852dc;

{$endif}




  // grcp437    Checksum: 35367

  _grcp437 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 
    064, 066, 068, 070, 072, 074, 076, 078, 080, 082, 084, 086, 088, 090, 092, 094, 
    096, 098, 100, 102, 104, 106, 108, 110, 112, 114, 116, 182, 183, 184, 185, 186, 
    187, 065, 067, 069, 071, 073, 075, 077, 079, 081, 083, 085, 087, 089, 091, 093, 
    095, 097, 099, 101, 103, 105, 107, 109, 111, 113, 115, 188, 189, 190, 191, 192, 
    117, 119, 120, 121, 122, 124, 125, 127, 128, 130, 131, 132, 133, 134, 135, 137, 
    138, 139, 140, 141, 143, 144, 145, 146, 148, 150, 151, 152, 153, 155, 156, 158, 
    159, 162, 163, 164, 165, 166, 167, 169, 170, 171, 172, 173, 174, 177, 178, 179, 
    193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 
    209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 
    225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 
    180, 149, 154, 157, 160, 161, 168, 176, 175, 181, 118, 123, 126, 129, 136, 142, 
    147, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );

  db437gr0 :PCollationTable = @_grcp437;




  // dbHebrew      Checksum: 43413

  _dbHebrew :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 
    064, 065, 066, 067, 068, 069, 070, 071, 072, 073, 074, 075, 076, 077, 078, 079, 
    080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 090, 091, 092, 093, 094, 095, 
    096, 097, 098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 
    112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 
    128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 
    144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 
    160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 
    176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 
    192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 
    208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 
    224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 
    240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255
  );
  dbhebrew :PCollationTable = @_dbhebrew;





  // slovene    Checksum: 48585

  _slovene :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015, 
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031, 
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047, 
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063, 
    064, 071, 078, 081, 089, 098, 104, 106, 108, 112, 116, 118, 123, 128, 132, 140, 
    146, 148, 152, 159, 167, 175, 181, 183, 185, 188, 193, 198, 199, 200, 201, 202, 
    203, 065, 077, 079, 087, 093, 103, 105, 107, 109, 115, 117, 119, 127, 129, 135, 
    145, 147, 149, 155, 164, 170, 180, 182, 184, 186, 190, 204, 205, 206, 207, 208, 
    082, 171, 095, 068, 066, 174, 085, 080, 121, 094, 144, 139, 111, 194, 072, 086, 
    100, 124, 120, 138, 136, 126, 122, 160, 156, 141, 176, 168, 165, 125, 209, 083, 
    067, 110, 137, 172, 076, 070, 197, 196, 102, 097, 210, 191, 084, 157, 211, 212, 
    213, 214, 215, 216, 217, 073, 074, 101, 161, 218, 219, 220, 221, 195, 192, 222, 
    223, 224, 225, 226, 227, 228, 075, 069, 229, 230, 231, 232, 233, 234, 235, 236, 
    091, 092, 090, 099, 088, 134, 113, 114, 096, 237, 238, 239, 240, 169, 179, 241, 
    142, 158, 143, 133, 130, 131, 163, 162, 153, 177, 150, 178, 187, 189, 166, 242, 
    243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 173, 154, 151, 254, 255
  );
  db852sl0 :PCollationTable = @_slovene;

{$ifdef USE_PARADOX_COLLATIONS}
  grcp437 :PCollationTable = @_grcp437;

  hebrew :PCollationTable = @_dbhebrew;

  slovene :PCollationTable = @_slovene;
{$endif}



  {$IFDEF PARADOX_COLLATIONS}

  // turk    8582

  _turk :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 072, 082, 084, 088, 094, 100, 102, 104, 109, 116, 118, 120, 122, 125, 132,
    140, 142, 144, 146, 149, 154, 161, 163, 165, 168, 170, 171, 172, 173, 174, 175,
    176, 065, 081, 083, 087, 089, 099, 101, 103, 114, 115, 117, 119, 121, 123, 127,
    139, 141, 143, 145, 148, 150, 160, 162, 164, 166, 169, 177, 178, 179, 180, 035,
    035, 035, 035, 099, 035, 035, 035, 035, 035, 035, 146, 035, 132, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 145, 035, 127, 035, 035, 168,
    216, 189, 193, 182, 195, 194, 198, 207, 211, 192, 197, 190, 186, 203, 185, 201,
    210, 204, 215, 214, 202, 199, 206, 212, 209, 213, 196, 191, 188, 187, 205, 184,
    074, 076, 075, 077, 080, 073, 078, 086, 098, 095, 096, 097, 113, 110, 111, 112,
    088, 126, 135, 133, 134, 136, 138, 200, 183, 157, 155, 156, 159, 168, 035, 147,
    067, 069, 068, 070, 079, 066, 071, 085, 093, 090, 091, 092, 108, 107, 106, 105,
    087, 124, 129, 128, 130, 131, 137, 208, 181, 152, 153, 151, 158, 166, 035, 167
  );
  turk :PCollationTable = @_turk;

  {$ENDIF}




  {$IFDEF PARADOX_COLLATIONS}

  // anczech    44872

  _anczech :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 078, 080, 088, 094, 104, 106, 108, 110, 116, 118, 120, 128, 130, 136,
    146, 148, 150, 156, 165, 171, 181, 183, 185, 187, 191, 198, 199, 200, 201, 202,
    203, 065, 077, 079, 087, 093, 103, 105, 107, 109, 115, 117, 119, 127, 129, 135,
    145, 147, 149, 155, 164, 170, 180, 182, 184, 186, 190, 204, 205, 206, 207, 208,
    250, 251, 209, 252, 210, 211, 212, 213, 253, 214, 163, 215, 158, 167, 197, 195,
    254, 216, 217, 218, 219, 220, 221, 222, 255, 223, 162, 224, 157, 166, 196, 194,
    225, 226, 227, 122, 228, 074, 229, 230, 231, 232, 160, 233, 234, 235, 236, 193,
    237, 238, 239, 121, 240, 241, 242, 243, 244, 073, 159, 245, 126, 246, 125, 192,
    152, 068, 070, 072, 076, 124, 082, 084, 086, 096, 100, 102, 098, 112, 114, 092,
    090, 132, 134, 138, 140, 144, 142, 247, 154, 175, 173, 179, 177, 189, 169, 161,
    151, 067, 069, 071, 075, 123, 081, 083, 085, 095, 099, 101, 097, 111, 113, 091,
    089, 131, 133, 137, 139, 143, 141, 248, 153, 174, 172, 178, 176, 188, 168, 249
  );
  anczech :PCollationTable = @_anczech;

  anczechw :PCollationTable = @_anczech;

  anil2czw :PCollationTable = @_anczech;

  {$ENDIF}



  {$IFDEF PARADOX_COLLATIONS}

  // cskamenw    40577

  _cskamenw :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 072, 074, 076, 078, 082, 084, 086, 088, 092, 094, 096, 098, 100, 102,
    110, 112, 114, 116, 120, 122, 128, 130, 132, 134, 138, 139, 140, 141, 142, 143,
    144, 065, 071, 073, 075, 077, 081, 083, 085, 087, 091, 093, 095, 097, 099, 101,
    109, 111, 113, 115, 119, 121, 127, 129, 131, 133, 137, 145, 146, 147, 148, 149,
    035, 035, 035, 081, 035, 035, 035, 035, 035, 035, 118, 035, 102, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 117, 035, 101, 035, 035, 134,
    161, 035, 035, 035, 035, 035, 035, 151, 035, 035, 035, 152, 035, 035, 035, 035,
    158, 156, 160, 035, 035, 155, 035, 159, 035, 035, 035, 153, 150, 035, 035, 035,
    066, 068, 066, 066, 070, 066, 066, 074, 078, 080, 078, 078, 088, 090, 088, 088,
    076, 100, 102, 104, 106, 102, 108, 035, 102, 122, 124, 122, 126, 136, 035, 154,
    065, 067, 065, 065, 069, 065, 065, 073, 077, 079, 077, 077, 087, 089, 087, 087,
    075, 099, 101, 103, 105, 101, 107, 157, 101, 121, 123, 121, 125, 135, 035, 133
  );
  cskamenw :PCollationTable = @_cskamenw;

  cskamen :PCollationTable = @_cskamenw;

  {$ENDIF}




  {$IFDEF PARADOX_COLLATIONS}

  // anpolish    44922

  _anpolish :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 078, 080, 088, 094, 104, 106, 108, 110, 116, 118, 120, 128, 130, 136,
    146, 148, 150, 156, 165, 171, 181, 183, 185, 187, 191, 198, 199, 200, 201, 202,
    203, 065, 077, 079, 087, 093, 103, 105, 107, 109, 115, 117, 119, 127, 129, 135,
    145, 147, 149, 155, 164, 170, 180, 182, 184, 186, 190, 204, 205, 206, 207, 208,
    250, 251, 237, 252, 229, 230, 231, 233, 253, 234, 162, 236, 158, 167, 197, 193,
    254, 217, 222, 223, 224, 225, 226, 227, 255, 232, 161, 249, 157, 166, 196, 192,
    210, 241, 242, 122, 228, 074, 218, 243, 247, 220, 160, 211, 235, 238, 219, 195,
    246, 221, 240, 121, 213, 214, 215, 216, 245, 073, 159, 212, 126, 239, 125, 194,
    152, 068, 070, 072, 076, 124, 082, 084, 086, 096, 100, 102, 098, 112, 114, 092,
    090, 132, 134, 138, 140, 144, 142, 209, 154, 175, 173, 179, 177, 189, 169, 163,
    151, 067, 069, 071, 075, 123, 081, 083, 085, 095, 099, 101, 097, 111, 113, 091,
    089, 131, 133, 137, 139, 143, 141, 244, 153, 174, 172, 178, 176, 188, 168, 248
  );
  anpolish :PCollationTable = @_anpolish;

  {$ENDIF}




  {$IFDEF PARADOX_COLLATIONS}

  // ancyrr    15158

  _ancyrr :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 068, 070, 072, 074, 076, 078, 080, 082, 084, 086, 088, 090, 092, 094,
    096, 098, 100, 102, 104, 106, 108, 110, 112, 114, 116, 183, 184, 185, 186, 187,
    188, 065, 067, 069, 071, 073, 075, 077, 079, 081, 083, 085, 087, 089, 091, 093,
    095, 097, 099, 101, 103, 105, 107, 109, 111, 113, 115, 189, 190, 191, 192, 193,
    194, 195, 196, 197, 198, 199, 200, 201, 254, 202, 203, 204, 205, 206, 207, 208,
    209, 210, 211, 212, 213, 214, 215, 216, 255, 217, 218, 219, 220, 221, 222, 223,
    224, 225, 226, 227, 228, 229, 230, 231, 130, 232, 233, 234, 235, 236, 237, 238,
    239, 240, 241, 242, 243, 244, 245, 246, 129, 247, 248, 249, 250, 251, 252, 253,
    118, 120, 122, 124, 126, 128, 132, 134, 136, 138, 140, 142, 144, 146, 148, 150,
    152, 154, 156, 158, 160, 162, 164, 166, 168, 170, 172, 174, 176, 178, 180, 182,
    117, 119, 121, 123, 125, 127, 131, 133, 135, 137, 139, 141, 143, 145, 147, 149,
    151, 153, 155, 157, 159, 161, 163, 165, 167, 169, 171, 173, 175, 177, 179, 181
  );
  ancyrr :PCollationTable = @_ancyrr;

  {$ENDIF}




  {$IFDEF PARADOX_COLLATIONS}

  // anhundc    19132

  _anhundc :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 071, 073, 075, 077, 081, 083, 085, 087, 092, 094, 096, 098, 100, 102,
    113, 115, 117, 119, 121, 123, 134, 136, 138, 140, 142, 143, 144, 145, 146, 147,
    148, 065, 070, 072, 074, 076, 080, 082, 084, 086, 091, 093, 095, 097, 099, 101,
    112, 114, 116, 118, 120, 122, 133, 135, 137, 139, 141, 149, 150, 151, 152, 153,
    250, 251, 154, 252, 155, 156, 157, 158, 253, 159, 160, 161, 126, 162, 163, 090,
    254, 164, 165, 166, 167, 168, 169, 170, 255, 171, 172, 173, 132, 174, 111, 175,
    176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191,
    192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 105, 204, 130, 205,
    206, 068, 207, 208, 209, 210, 069, 211, 212, 079, 213, 214, 215, 089, 216, 217,
    218, 219, 220, 104, 221, 110, 107, 222, 223, 224, 125, 131, 128, 225, 226, 227,
    228, 067, 229, 230, 231, 232, 233, 234, 235, 078, 236, 237, 238, 088, 239, 240,
    241, 242, 243, 103, 109, 108, 106, 244, 245, 246, 124, 129, 127, 247, 248, 249
  );
  anhundc :PCollationTable = @_anhundc;

  {$ENDIF}



  {$IFDEF PARADOX_COLLATIONS}

  // angreek1    39126

  _angreek1 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 066, 068, 070, 072, 074, 076, 078, 080, 082, 084, 086, 088, 090, 092, 094,
    096, 098, 100, 102, 104, 106, 108, 110, 112, 114, 116, 186, 187, 188, 189, 190,
    191, 065, 067, 069, 071, 073, 075, 077, 079, 081, 083, 085, 087, 089, 091, 093,
    095, 097, 099, 101, 103, 105, 107, 109, 111, 113, 115, 192, 193, 194, 195, 196,
    238, 239, 197, 198, 199, 200, 201, 202, 240, 203, 241, 204, 242, 243, 244, 245,
    246, 205, 206, 207, 208, 209, 210, 211, 247, 212, 248, 213, 249, 250, 251, 252,
    214, 215, 120, 216, 217, 218, 219, 220, 221, 222, 253, 223, 224, 225, 226, 227,
    228, 229, 230, 231, 232, 233, 234, 235, 130, 136, 144, 236, 159, 237, 174, 185,
    142, 119, 122, 124, 126, 129, 132, 135, 138, 143, 147, 149, 151, 153, 155, 158,
    161, 163, 254, 166, 168, 173, 177, 179, 181, 184, 145, 175, 118, 128, 134, 140,
    172, 117, 121, 123, 125, 127, 131, 133, 137, 139, 146, 148, 150, 152, 154, 156,
    160, 162, 165, 164, 167, 169, 176, 178, 180, 182, 141, 171, 157, 170, 183, 255
  );
  angreek1 :PCollationTable = @_angreek1;

  ACCGREEK :PCollationTable = @_angreek1;

  {$ENDIF}

  {$IFDEF PARADOX_COLLATIONS}

  // ansislov    61480

  _ansislov :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 071, 078, 081, 089, 098, 104, 106, 108, 112, 116, 118, 123, 128, 132, 140,
    146, 148, 152, 159, 167, 175, 181, 183, 185, 188, 193, 198, 199, 200, 201, 202,
    203, 065, 077, 079, 087, 093, 103, 105, 107, 109, 115, 117, 119, 127, 129, 135,
    145, 147, 149, 155, 164, 170, 180, 182, 184, 186, 190, 204, 205, 206, 207, 208,
    250, 251, 209, 252, 210, 211, 212, 213, 253, 214, 163, 215, 160, 168, 197, 194,
    254, 216, 217, 218, 219, 220, 221, 222, 255, 223, 162, 224, 156, 165, 196, 191,
    225, 226, 227, 125, 228, 076, 229, 230, 231, 232, 161, 233, 234, 235, 236, 195,
    237, 238, 239, 121, 240, 241, 242, 243, 244, 070, 157, 245, 126, 246, 122, 192,
    153, 073, 074, 075, 072, 124, 086, 082, 084, 100, 102, 099, 101, 113, 114, 090,
    092, 133, 134, 142, 143, 144, 141, 247, 154, 179, 177, 178, 176, 189, 169, 158,
    150, 067, 068, 069, 066, 120, 085, 080, 083, 095, 097, 094, 096, 110, 111, 088,
    091, 130, 131, 137, 138, 139, 136, 248, 151, 174, 172, 173, 171, 187, 166, 249
  );
  ansislov :PCollationTable = @_ansislov;

  {$ENDIF}



  {$IFDEF USE_PARADOX_COLLATIONS}

  // ANTURK    24004

  _ANTURK :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 072, 082, 084, 088, 094, 100, 102, 106, 112, 120, 122, 124, 126, 129, 136,
    144, 146, 148, 150, 155, 160, 167, 169, 171, 174, 176, 177, 178, 179, 180, 181,
    182, 065, 081, 083, 087, 089, 099, 101, 105, 117, 119, 121, 123, 125, 127, 131,
    143, 145, 147, 149, 154, 156, 166, 168, 170, 172, 175, 183, 184, 185, 186, 187,
    248, 249, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 250, 251, 252,
    253, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 254, 255, 211,
    212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227,
    228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243,
    074, 076, 075, 077, 080, 073, 078, 086, 098, 095, 096, 097, 116, 113, 114, 115,
    104, 130, 139, 137, 138, 140, 142, 244, 245, 163, 161, 162, 165, 118, 153, 151,
    067, 069, 068, 070, 079, 066, 071, 085, 093, 090, 091, 092, 111, 110, 109, 108,
    103, 128, 133, 132, 134, 135, 141, 246, 247, 158, 159, 157, 164, 107, 152, 173
  );
  ANTURK :PCollationTable = @_ANTURK;

  {$ENDIF}



  // DB857TR0      Checksum: 20238

  _DB857TR0 :TCollationTable = (
    000, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 
    175, 176, 177, 178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 
    001, 141, 154, 147, 148, 133, 146, 156, 122, 123, 134, 131, 136, 132, 135, 142, 
    002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 138, 137, 128, 130, 129, 140, 
    145, 012, 020, 021, 023, 024, 029, 030, 032, 033, 039, 040, 041, 042, 043, 045, 
    051, 052, 053, 054, 056, 057, 062, 063, 064, 065, 066, 124, 143, 125, 157, 159, 
    155, 067, 074, 075, 077, 078, 083, 084, 086, 092, 093, 094, 095, 096, 097, 099, 
    105, 106, 107, 108, 110, 111, 116, 117, 118, 119, 121, 126, 144, 127, 158, 191, 
    022, 115, 079, 071, 072, 069, 073, 076, 081, 082, 080, 091, 089, 087, 013, 014, 
    028, 152, 019, 103, 104, 102, 114, 113, 038, 050, 061, 149, 150, 151, 055, 109, 
    068, 088, 100, 112, 098, 044, 031, 085, 139, 192, 193, 194, 195, 196, 197, 198, 
    199, 200, 201, 202, 203, 015, 016, 018, 204, 205, 206, 207, 208, 209, 210, 211, 
    212, 213, 214, 215, 216, 217, 070, 017, 218, 219, 220, 221, 222, 223, 224, 225, 
    226, 227, 025, 026, 027, 253, 034, 035, 037, 228, 229, 230, 231, 232, 036, 233, 
    046, 153, 047, 049, 101, 048, 234, 254, 235, 058, 059, 060, 090, 120, 236, 237, 
    238, 239, 255, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252
  );
  DB857TR0 :PCollationTable = @_DB857TR0;




  // FOXCZWIN    29353

  _FOXCZWIN :TCollationTable = (
    000, 001, 001, 001, 001, 001, 001, 001, 001, 002, 003, 004, 005, 006, 001, 001,
    001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001, 001,
    016, 007, 008, 009, 010, 011, 012, 013, 014, 015, 016, 017, 018, 019, 020, 021,
    080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 022, 023, 024, 025, 026, 027,
    028, 090, 097, 098, 102, 105, 110, 111, 112, 113, 116, 117, 118, 122, 123, 126,
    131, 132, 133, 136, 141, 144, 150, 151, 152, 153, 155, 029, 030, 031, 032, 033,
    034, 090, 097, 098, 102, 105, 110, 111, 112, 113, 116, 117, 118, 122, 123, 126,
    131, 132, 133, 136, 141, 144, 150, 151, 152, 153, 155, 035, 036, 037, 038, 001,
    159, 160, 039, 161, 040, 041, 042, 043, 162, 044, 140, 045, 137, 142, 158, 157,
    163, 046, 047, 048, 049, 050, 051, 052, 164, 053, 140, 054, 137, 142, 158, 157,
    055, 056, 057, 121, 058, 095, 059, 060, 061, 062, 138, 063, 064, 065, 066, 156,
    067, 068, 069, 121, 070, 071, 072, 073, 074, 096, 138, 075, 120, 076, 120, 156,
    134, 092, 093, 094, 091, 119, 099, 100, 101, 107, 109, 106, 108, 114, 115, 103,
    104, 124, 125, 128, 130, 129, 127, 077, 135, 146, 145, 147, 145, 154, 143, 139,
    134, 092, 093, 094, 091, 119, 099, 100, 101, 107, 109, 106, 108, 114, 115, 103,
    104, 124, 125, 128, 130, 129, 127, 078, 135, 147, 146, 148, 149, 154, 143, 079
  );
  FOXCZWIN :PCollationTable = @_FOXCZWIN;




  // FOXCZ895      Checksum: 42384

  _FOXCZ895 :TCollationTable = (
    000, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 
    032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 032, 
    048, 036, 037, 038, 039, 040, 041, 033, 042, 043, 044, 063, 045, 034, 046, 048, 
    131, 133, 134, 136, 137, 138, 139, 140, 141, 142, 049, 050, 065, 066, 067, 051, 
    052, 145, 152, 154, 158, 162, 168, 170, 172, 174, 178, 180, 182, 188, 191, 195, 
    203, 205, 207, 213, 218, 223, 231, 233, 235, 237, 241, 053, 054, 055, 056, 057, 
    058, 145, 152, 154, 158, 162, 168, 170, 172, 174, 178, 180, 182, 188, 191, 195, 
    203, 205, 207, 213, 218, 223, 231, 233, 235, 237, 241, 059, 060, 061, 062, 032, 
    156, 225, 164, 160, 147, 160, 220, 156, 166, 166, 184, 176, 186, 184, 147, 149, 
    164, 243, 243, 201, 197, 199, 229, 227, 239, 197, 225, 216, 186, 239, 211, 220, 
    149, 176, 199, 227, 193, 193, 229, 201, 216, 211, 209, 209, 132, 127, 069, 070, 
    124, 125, 126, 083, 105, 106, 150, 091, 090, 107, 084, 092, 100, 099, 098, 089, 
    093, 112, 108, 101, 081, 116, 102, 103, 096, 088, 115, 111, 104, 082, 119, 113, 
    114, 109, 110, 095, 094, 086, 087, 118, 117, 097, 085, 123, 121, 221, 122, 120, 
    244, 214, 245, 249, 251, 251, 129, 252, 254, 248, 255, 246, 143, 254, 247, 073, 
    075, 068, 077, 076, 078, 079, 071, 074, 128, 064, 130, 072, 191, 135, 080, 035
  );
  FOXCZ895 :PCollationTable = @_FOXCZ895;



  {$IFDEF USE_ACCESS_COLLATIONS}

  // BLROM800    28847

  _BLROM800 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    048, 049, 050, 051, 052, 053, 054, 055, 056, 057, 058, 059, 060, 061, 062, 063,
    064, 065, 066, 067, 068, 069, 070, 071, 072, 073, 074, 075, 076, 077, 078, 079,
    080, 081, 082, 083, 084, 085, 086, 087, 088, 089, 090, 091, 092, 093, 094, 095,
    096, 097, 098, 099, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
    112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127,
    128, 129, 035, 161, 035, 035, 035, 035, 035, 035, 206, 035, 079, 130, 131, 132,
    133, 035, 035, 035, 035, 035, 035, 217, 035, 035, 207, 035, 111, 134, 135, 209,
    136, 155, 162, 158, 157, 159, 035, 160, 145, 035, 220, 222, 035, 035, 035, 035,
    150, 224, 035, 035, 144, 214, 215, 213, 035, 035, 221, 223, 218, 219, 216, 156,
    137, 195, 138, 196, 187, 179, 182, 151, 139, 191, 140, 141, 201, 200, 142, 143,
    198, 153, 203, 202, 194, 204, 189, 035, 181, 146, 208, 147, 190, 148, 211, 193,
    171, 167, 163, 197, 175, 183, 186, 152, 172, 168, 164, 176, 188, 184, 180, 192,
    199, 154, 173, 169, 165, 205, 177, 035, 185, 174, 170, 166, 178, 149, 212, 210
  );
  BLROM800 :PCollationTable = @_BLROM800;

  {$ENDIF}


  {$IFDEF USE_ORACLE_COLLATIONS}

  // ORAWE850    31378

  _ORAWE850 :TCollationTable = ( 
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 048, 049, 050, 051, 052, 053,
    054, 101, 117, 119, 123, 127, 137, 139, 141, 143, 153, 155, 157, 159, 161, 165,
    179, 181, 183, 185, 188, 192, 202, 204, 206, 208, 213, 055, 056, 057, 058, 059,
    060, 102, 118, 120, 124, 128, 138, 140, 142, 144, 154, 156, 158, 160, 162, 166,
    180, 182, 184, 186, 189, 193, 203, 205, 207, 209, 214, 061, 062, 063, 064, 065,
    035, 035, 035, 068, 035, 035, 035, 035, 035, 035, 185, 035, 165, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 186, 035, 166, 035, 035, 208,
    100, 076, 080, 066, 082, 081, 083, 091, 095, 079, 069, 077, 073, 087, 072, 085,
    094, 088, 099, 098, 086, 084, 090, 096, 093, 097, 070, 078, 075, 074, 089, 071,
    105, 103, 107, 111, 109, 113, 115, 121, 131, 129, 133, 135, 147, 145, 149, 151,
    125, 163, 169, 167, 171, 175, 173, 067, 177, 196, 194, 198, 200, 210, 190, 187,
    106, 104, 108, 112, 110, 114, 116, 122, 132, 130, 134, 136, 148, 146, 150, 152,
    126, 164, 170, 168, 172, 176, 174, 092, 178, 197, 195, 199, 201, 211, 191, 212
  );
  ORAWE850 :PCollationTable = @_ORAWE850 ;

  {$ENDIF}



  {$IFDEF USE_SYBASE_COLLATIONS}

  // SYDC850    46023

  _SYDC850 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 048, 049, 050, 051, 052, 053,
    054, 111, 127, 129, 133, 135, 145, 147, 149, 151, 161, 163, 165, 167, 169, 173,
    187, 189, 191, 193, 196, 198, 208, 210, 212, 214, 219, 055, 056, 057, 058, 059,
    060, 112, 128, 130, 134, 136, 146, 148, 150, 152, 162, 164, 166, 168, 170, 174,
    188, 190, 192, 194, 197, 199, 209, 211, 213, 215, 220, 061, 062, 063, 064, 065,
    035, 035, 035, 068, 035, 035, 035, 035, 035, 035, 193, 035, 173, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 194, 035, 174, 035, 035, 214,
    100, 076, 080, 066, 082, 081, 083, 091, 095, 079, 069, 077, 073, 087, 072, 085,
    094, 088, 099, 098, 086, 084, 090, 096, 093, 097, 070, 078, 075, 074, 089, 071,
    113, 115, 117, 119, 121, 123, 125, 131, 137, 139, 141, 143, 153, 155, 157, 159,
    221, 171, 175, 177, 179, 181, 183, 067, 185, 200, 202, 204, 206, 216, 224, 195,
    114, 116, 118, 120, 122, 124, 126, 132, 138, 140, 142, 144, 154, 156, 158, 160,
    222, 172, 176, 178, 180, 182, 184, 092, 186, 201, 203, 205, 207, 217, 223, 218
  );
  SYDC850 :PCollationTable = @_SYDC850;




  // SYDC437    16660

  _SYDC437 :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 034, 035, 036, 037, 038, 039, 040, 041, 042, 043, 044, 045, 046, 047,
    085, 086, 087, 088, 089, 090, 091, 092, 093, 094, 048, 049, 050, 051, 052, 053,
    054, 095, 106, 108, 112, 114, 121, 123, 125, 127, 133, 135, 137, 139, 141, 145,
    152, 154, 156, 158, 160, 162, 169, 171, 173, 175, 178, 055, 056, 057, 058, 059,
    060, 096, 107, 109, 113, 115, 122, 124, 126, 128, 134, 136, 138, 140, 142, 146,
    153, 155, 157, 159, 161, 163, 170, 172, 174, 176, 179, 061, 062, 063, 064, 065,
    035, 035, 035, 069, 035, 035, 035, 035, 035, 035, 158, 035, 145, 035, 035, 035,
    035, 035, 035, 035, 035, 035, 035, 035, 035, 035, 159, 035, 146, 035, 035, 175,
    084, 076, 066, 067, 035, 068, 035, 035, 035, 035, 070, 077, 073, 035, 035, 035,
    081, 079, 083, 035, 035, 181, 035, 082, 035, 035, 071, 078, 075, 074, 035, 072,
    095, 095, 095, 095, 100, 102, 104, 110, 114, 117, 114, 114, 127, 127, 127, 127,
    112, 143, 145, 145, 145, 145, 150, 035, 145, 162, 162, 162, 167, 175, 035, 180,
    097, 098, 099, 096, 101, 103, 105, 111, 116, 118, 119, 120, 129, 130, 131, 132,
    113, 144, 147, 148, 149, 146, 151, 080, 146, 164, 165, 166, 168, 176, 035, 177
  );
  SYDC437 :PCollationTable = @_SYDC437;

  {$ENDIF}


  {$IFDEF USE_DB2_COLLATIONS}

  // db2andeu    8683

  _db2andeu :TCollationTable = (
    000, 001, 002, 003, 004, 005, 006, 007, 008, 009, 010, 011, 012, 013, 014, 015,
    016, 017, 018, 019, 020, 021, 022, 023, 024, 025, 026, 027, 028, 029, 030, 031,
    032, 033, 035, 038, 039, 045, 046, 047, 048, 049, 050, 051, 052, 053, 054, 055,
    056, 057, 058, 059, 060, 061, 062, 063, 064, 065, 066, 067, 068, 069, 070, 071,
    073, 074, 085, 087, 091, 094, 101, 103, 105, 107, 113, 115, 117, 119, 121, 125,
    132, 134, 136, 138, 141, 143, 150, 152, 154, 156, 159, 161, 162, 163, 164, 165,
    166, 075, 086, 088, 092, 095, 102, 104, 106, 108, 114, 116, 118, 120, 122, 126,
    133, 135, 137, 139, 142, 144, 151, 153, 155, 157, 160, 168, 169, 170, 171, 172,
    233, 234, 235, 041, 236, 237, 238, 239, 240, 241, 231, 242, 229, 243, 244, 245,
    246, 167, 211, 226, 227, 247, 248, 249, 250, 251, 230, 252, 228, 253, 254, 232,
    032, 034, 042, 040, 185, 043, 193, 217, 221, 182, 173, 036, 176, 213, 175, 210,
    220, 214, 225, 224, 212, 200, 216, 222, 219, 223, 174, 037, 178, 177, 215, 072,
    181, 179, 180, 184, 080, 081, 083, 089, 189, 100, 187, 188, 194, 190, 191, 192,
    093, 124, 197, 195, 196, 199, 130, 044, 202, 207, 205, 206, 148, 209, 204, 140,
    078, 084, 076, 183, 077, 079, 082, 090, 099, 096, 097, 098, 111, 112, 110, 109,
    186, 123, 129, 131, 127, 198, 128, 218, 201, 147, 149, 146, 145, 208, 203, 158
  );
  db2andeu :PCollationTable = @_db2andeu;

  {$ENDIF}

initialization

  InitialiseCollationTables;

  RegisterCollation( DbfLangId_Spanish_1252,    DBWINES0, 'DBWINES0' );         // 'Spanish' ANSI
  RegisterCollation( DbfLangId_Ascii_1252,      BINARY_COLLATION, 'DBWINUS0' ); // 'ascii' ANSI
  RegisterCollation( DbfLangId_WEurope_1252,    DBWINWE0, 'DBWINWE0' );         // 'WEurope' ANSI

  //DbfLangId_BUL_868       = $8E;    // is it used? does not exist in BDE
  //DbfLangId_FIN_850       = $0C;    // is it used? does not exist in BDE
  RegisterCollation( DbfLangId_DEU_437,     DB437DE0, 'DB437DE0' ); // dBASE DEU cp437
  RegisterCollation( DbfLangId_ESP_437,     DB437ES1, 'DB437ES1' ); // dBASE ESP cp437
  RegisterCollation( DbfLangId_FIN_437,     DB437FI0, 'DB437FI0' ); // dBASE FIN cp437
  RegisterCollation( DbfLangId_FRA_437,     DB437FR0, 'DB437FR0' ); // dBASE FRA cp437
  RegisterCollation( DbfLangId_ELL_437,     db437gr0, 'db437gr0' ); // dBASE ELL GR437
  RegisterCollation( DbfLangId_ITA_437,     DB437IT0, 'DB437IT0' ); // dBASE ITA cp437
  RegisterCollation( DbfLangId_NLD_437,     DB437NL0, 'DB437NL0' ); // dBASE NLD cp437
  RegisterCollation( DbfLangId_SVE_437,     DB437SV0, 'DB437SV0' ); // dBASE SVE cp437
  RegisterCollation( DbfLangId_ENG_437,     DB437UK0, 'DB437UK0' ); // dBASE ENG cp437
  RegisterCollation( DbfLangId_ENU_437,     DB437US0, 'DB437US0' ); // dBASE ENU cp437
  RegisterCollation( DbfLangId_FRC_850,     DB850CF0, 'DB850CF0' ); // dBASE FRC cp850
  RegisterCollation( DbfLangId_DEU_850,     DB850DE0, 'DB850DE0' ); // dBASE DEU cp850
  RegisterCollation( DbfLangId_ESP_850,     DB850ES0, 'DB850ES0' ); // dBASE ESP cp850
  RegisterCollation( DbfLangId_FRA_850,     DB850FR0, 'DB850FR0' ); // dBASE FRA cp850
  RegisterCollation( DbfLangId_ITA_850,     DB850IT1, 'DB850IT1' ); // dBASE ITA cp850
  RegisterCollation( DbfLangId_NLD_850,     DB850NL0, 'DB850NL0' ); // dBASE NLD cp850
  RegisterCollation( DbfLangId_PTB_850,     DB850PT0, 'DB850PT0' ); // dBASE PTB cp850
  RegisterCollation( DbfLangId_SVE_850,     DB850SV1, 'DB850SV1' ); // dBASE SVE cp850
  RegisterCollation( DbfLangId_ENG_850,     DB850UK0, 'DB850UK0' ); // dBASE ENG cp850
  RegisterCollation( DbfLangId_ENU_850,     DB850US0, 'DB850US0' ); // dBASE ENU cp850
  RegisterCollation( DbfLangId_CSY_852,     DB852CZ0, 'DB852CZ0' ); // dBASE CSY cp852
  RegisterCollation( DbfLangId_HUN_852,     db852hdc, 'db852hdc' ); // dBASE HUN cp852
  RegisterCollation( DbfLangId_PLK_852,     db852po0, 'db852po0' ); // dBASE PLK cp852
  RegisterCollation( DbfLangId_SLO_852,     db852sl0, 'db852sl0' ); // dBASE SLO cp852
  RegisterCollation( DbfLangId_TRK_857,     DB857TR0, 'DB857TR0' ); // dBASE TRK cp857
  RegisterCollation( DbfLangId_PTG_860,     DB860PT0, 'DB860PT0' ); // dBASE PTG cp860
  RegisterCollation( DbfLangId_FRC_863,     DB863CF1, 'DB863CF1' ); // dBASE FRC cp863
  RegisterCollation( DbfLangId_DAN_865,     DB865DA0, 'DB865DA0' ); // dBASE DAN cp865
  RegisterCollation( DbfLangId_NOR_865,     DB865NO0, 'DB865NO0' ); // dBASE NOR cp865
  RegisterCollation( DbfLangId_RUS_866,     db866ru0, 'db866ru0' ); // dBASE RUS cp866
  RegisterCollation( DbfLangId_CSY_867,     DB867CZ0, 'DB867CZ0' ); // dBASE CSY cp867
  RegisterCollation( DbfLangId_THA_874,     db874th0, 'db874th0' ); // dBASE THA cp874
  RegisterCollation( DbfLangId_JPN_932,     BINARY_COLLATION, 'DB932JP0' ); // dBASE JPN cp932
  RegisterCollation( DbfLangId_JPN_DIC_932, DB932JP1, 'DB932JP1' ); // dBASE JPN Dic932
  RegisterCollation( DbfLangId_CHS_936,     DB936CN0, 'DB936CN0' ); // dBASE CHS cp936
  RegisterCollation( DbfLangId_KOR_949,     DB949KO0, 'DB949KO0' ); // dBASE KOR cp949
  RegisterCollation( DbfLangId_CHT_950,     DB950TW0, 'DB950TW0' ); // dBASE CHT cp950
  RegisterCollation( DbfLangId_Hebrew,      dbHebrew, 'dbHebrew' ); // Hebrew dBASE
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1CA0 ); // Borland FRC Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1DA0 ); // Borland DAN Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1DE0 ); // Borland DEU Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1ES0 ); // Borland ESP Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1FI0 ); // Borland FIN Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1FR0 ); // Borland FRA Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1IS0 ); // Borland ISL Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1IT0 ); // Borland ITA Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1NL0 ); // Borland NLD Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1NO0 ); // Borland NOR Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1PT0 ); // Borland PTG Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1SV0 ); // Borland SVE Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1UK0 ); // Borland ENG Latin-1
  //RegisterCollation( XXXXXXXXXXXX,    BLLT1US0 ); // Borland ENU Latin-1

(*

  FoxLangId_ENU_437       = $01;
  FoxLangId_Intl_850      = $02;
  FoxLangId_Windows_1252  = $03;
  FoxLangId_Mac_10000     = $04;

  FoxLangId_EEurope_852   = $64;
  FoxLangId_Russia_866    = $65;
  FoxLangId_Nordic_865    = $66;
  FoxLangId_Iceland_861   = $67;

  DbfLangId_POL_620       = $69;

  FoxLangId_Greek_737     = $6A;
  FoxLangId_Turkish_857   = $6B;

  FoxLangId_Taiwan_950    = $78;
  FoxLangId_Korean_949    = $79;
  FoxLangId_Chinese_936   = $7A;
  FoxLangId_Japan_932     = $7B;
  FoxLangId_Thai_874      = $7C;
  FoxLangId_Hebrew_1255   = $7D;
  FoxLangId_Arabic_1256   = $7E;

  FoxLangId_Russia_10007  = $96;
  FoxLangId_EEurope_10029 = $97;
  FoxLangId_Greek_10006   = $98;

  FoxLangId_Czech_850     = $9C;    // DOS

  FoxLangId_EEurope_1250  = $C8;
  FoxLangId_Russia_1251   = $C9;
  FoxLangId_Turkish_1254  = $CA;
  FoxLangId_Greek_1253    = $CB;

*)


  RegisterCollation( FoxLangId_Czech_895,     FOXCZ895, 'FOXCZ895' ); // FoxPro Czech DOS895
  RegisterCollation( FoxLangId_Czech_1250,    FOXCZWIN, 'FOXCZWIN' ); // FoxPro Czech 1250
  RegisterCollation( FoxLangId_German_437,    FOXDE437, 'FOXDE437' ); // FoxPro German 437
  RegisterCollation( FoxLangId_German_1252,   FOXDEWIN, 'FOXDEWIN' ); // FoxPro German 1252
  RegisterCollation( FoxLangId_Nordic_437,    FOXNO437, 'FOXNO437' ); // FoxPro Nordic 437
  RegisterCollation( FoxLangId_Nordic_850,    FOXNO850, 'FOXNO850' ); // FoxPro Nordic 850
  RegisterCollation( FoxLangId_Nordic_1252,   FOXNOWIN, 'FOXNOWIN' ); // FoxPro Nordic 1252

  //RegisterCollation( XXXXXXXXXXXX,    BLROM800 ); // SQL Link ROMAN8
  //RegisterCollation( XXXXXXXXXXXX,    db2andeu ); // DB2 SQL ANSI DEU
  //RegisterCollation( XXXXXXXXXXXX,    SYDC437 );  // Sybase SQL Dic437
  //RegisterCollation( XXXXXXXXXXXX,    SYDC850 );  // Sybase SQL Dic850
  //RegisterCollation( XXXXXXXXXXXX,    ORAWE850 ); // Oracle SQL WE850

  //RegisterCollation( XXXXXXXXXXXX,    ACCGEN );   // Access General
  //RegisterCollation( XXXXXXXXXXXX,    ACCGREEK ); // Access Greece
  //RegisterCollation( XXXXXXXXXXXX,    ACCJAPAN ); // Access Japanese
  //RegisterCollation( XXXXXXXXXXXX,    ACCNRDAN ); // Access Nord/Danish
  //RegisterCollation( XXXXXXXXXXXX,    ACCSWFIN ); // Access Swed/Finnish

  //RegisterCollation( XXXXXXXXXXXX,    ancyrr );   // Pdox ANSI Cyrillic
  //RegisterCollation( XXXXXXXXXXXX,    anczech );  // Pdox ANSI Czech
  //RegisterCollation( XXXXXXXXXXXX,    anczechw ); // pdx ANSI Czech 'CH'
  //RegisterCollation( XXXXXXXXXXXX,    angreek1 ); // Pdox ANSI Greek
  //RegisterCollation( XXXXXXXXXXXX,    ANHEBREW ); // Paradox ANSI HEBREW
  //RegisterCollation( XXXXXXXXXXXX,    anhundc );  // Pdox ANSI Hun. DC
  //RegisterCollation( XXXXXXXXXXXX,    anil2czw ); // pdx ANSI ISO L_2 CZ
  //RegisterCollation( XXXXXXXXXXXX,    anpolish ); // Pdox ANSI Polish
  //RegisterCollation( XXXXXXXXXXXX,    ANSII850 ); // Pdox ANSI Intl850
  //RegisterCollation( XXXXXXXXXXXX,    ANSIINTL ); // Pdox ANSI Intl
  //RegisterCollation( XXXXXXXXXXXX,    ANSINOR4 ); // Pdox ANSI Nordan4
  //RegisterCollation( XXXXXXXXXXXX,    ansislov ); // Pdox ANSI Slovene
  //RegisterCollation( XXXXXXXXXXXX,    ANSISPAN ); // Pdox ANSI Spanish
  //RegisterCollation( XXXXXXXXXXXX,    ANSISWFN ); // Pdox ANSI Swedfin
  //RegisterCollation( XXXXXXXXXXXX,    ANTURK );   // Pdox ANSI Turkish
  //RegisterCollation( XXXXXXXXXXXX,    china );    // Paradox China 936
  //RegisterCollation( XXXXXXXXXXXX,    cskamen );  // Paradox Czech 867
  //RegisterCollation( XXXXXXXXXXXX,    cskamenw ); // pdx Czech 867 'CH'
  //RegisterCollation( XXXXXXXXXXXX,    cyrr );     // Paradox Cyrr 866
  //RegisterCollation( XXXXXXXXXXXX,    czech );    // Paradox Czech 852
  //RegisterCollation( XXXXXXXXXXXX,    czechw );   // pdx Czech 852 'CH'
  //RegisterCollation( XXXXXXXXXXXX,    grcp437 );  // Paradox Greek GR437
  //RegisterCollation( XXXXXXXXXXXX,    hebrew );   // Paradox 'hebrew'
  //RegisterCollation( XXXXXXXXXXXX,    hun852dc ); // Paradox Hun 852 DC
  //RegisterCollation( XXXXXXXXXXXX,    iceland );  // Paradox ISL 861
  //RegisterCollation( XXXXXXXXXXXX,    il2czw );   // pdx ISO L_2 Czech
  //RegisterCollation( XXXXXXXXXXXX,    intl );     // Paradox 'intl'
  //RegisterCollation( XXXXXXXXXXXX,    intl850 );  // Paradox 'intl' 850
  //RegisterCollation( XXXXXXXXXXXX,    japan );    // Paradox 'japan'
  //RegisterCollation( XXXXXXXXXXXX,    korea );    // Paradox Korea 949
  //RegisterCollation( XXXXXXXXXXXX,    nordan );   // Paradox 'nordan'
  //RegisterCollation( XXXXXXXXXXXX,    nordan40 ); // Paradox 'nordan40'
  //RegisterCollation( XXXXXXXXXXXX,    polish );   // Paradox Polish 852
  //RegisterCollation( XXXXXXXXXXXX,    slovene );  // Paradox Slovene 852
  //RegisterCollation( XXXXXXXXXXXX,    SPANISH );  // Paradox ESP 437
  //RegisterCollation( XXXXXXXXXXXX,    swedfin );  // Paradox 'swedfin'
  //RegisterCollation( XXXXXXXXXXXX,    taiwan );   // Paradox Taiwan 950
  //RegisterCollation( XXXXXXXXXXXX,    thai );     // Paradox Thai 874
  //RegisterCollation( XXXXXXXXXXXX,    turk );     // Paradox 'turk'

end.

