
{
    This file is part of the Free Component Library (FCL)
    Copyright (c) 2023 by the Free Pascal development team

    See the file COPYING.FPC, included in this distribution,
    for details about the copyright.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

 **********************************************************************}
{
  Blowfish II 128-bit block 2005 by Alexander Pukall
  Original C code can be found on
  http://pccipher.free.fr/blowfish2/blowfish2.txt
}

{$ifdef fpc}
{$mode objfpc}
{$h+}
{$inline on}
{$endif}

{$IFNDEF FPC_DOTTEDUNITS}
unit BlowFish2;
{$ENDIF FPC_DOTTEDUNITS}

interface

{$IFDEF FPC_DOTTEDUNITS}
uses System.SysUtils, System.Classes;
{$ELSE FPC_DOTTEDUNITS}
uses SysUtils,Classes;
{$ENDIF FPC_DOTTEDUNITS}


Type
  TBlowFishPArray = Array[0..(64 + 2)-1] of QWord;
  TBlowFishSArray = Array[0..(8-1)] of array [0..(256-1)] of QWord;
  TBlowfish2Context = record
   P : TBlowFishPArray;
   S : TBlowFishSArray;
  end;
  PBlowfish2Context = ^TBlowfish2Context;

  TBF2Block     = array[0..1] of QWord;     { BlowFish2 }


procedure Blowfish2_Init(ctx : PBlowfish2Context; key : pansichar; keyLen : Integer);
procedure Blowfish2_Init(ctx : PBlowfish2Context; key : TBytes);

procedure Blowfish2_Encrypt(ctx : PBlowfish2Context; var xl, xr: QWord);
procedure Blowfish2_Encrypt(ctx : PBlowfish2Context; var aBlock : TBF2Block); inline;

procedure Blowfish2_Decrypt(ctx : PBlowfish2Context; var xl, xr: QWord);
procedure Blowfish2_Decrypt(ctx : PBlowfish2Context; var aBlock : TBF2Block); inline;

Type
  EBlowFish2Error = Class(EStreamError);

  { TBlowFishStream }

  { TBlowFish2Stream }

  TBlowFish2Stream = Class(TOwnerStream)
  Private
    FBF     : TBlowfish2Context;
    FData   : TBF2Block;
    FBufpos : Byte;
    FPos    : Int64;
  protected
    function GetPosition: Int64; override;
    procedure InvalidSeek; override;
  Public
    Constructor Create(const AKey : UTF8String; Dest: TStream); overload; virtual;
    Constructor Create(AKey : TBytes; Dest: TStream); overload; virtual;
    Destructor Destroy; override;
  end;

  TBlowFish2EncryptStream = Class(TBlowFish2Stream)
  public
    Destructor Destroy; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(const Offset: Int64; Origin: TSeekOrigin): Int64; override;
    procedure Flush;
  end;

  TBlowFish2DeCryptStream = Class(TBlowFish2Stream)
  private
    FSourcePos0: Int64;
  public
    Constructor Create(const AKey : UTF8String; Dest: TStream); overload; virtual;
    Constructor Create(AKey : TBytes; Dest: TStream); overload; virtual;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Seek(const Offset: Int64; Origin: TSeekOrigin): Int64; override;
  end;


Implementation

ResourceString
  SNoSeekAllowed  = 'Seek not allowed on encryption streams';
  SErrEmptyPassPhraseNotAllowed = 'Empty passphrase is not allowed in constructor';
  SErrKeyTooLong = 'Key length (%d) exceeds maximum (%d)';

Const
  N = 64;
  MaxKeyLen = 528;
  ORIG_P : TBlowFishPArray = (
        
QWord($243F6A8885A308D3),QWord($13198A2E03707344),QWord($A4093822299F31D0),
QWord($082EFA98EC4E6C89),QWord($452821E638D01377),QWord($BE5466CF34E90C6C),
QWord($C0AC29B7C97C50DD),QWord($3F84D5B5B5470917),QWord($9216D5D98979FB1B),
QWord($D1310BA698DFB5AC),QWord($2FFD72DBD01ADFB7),QWord($B8E1AFED6A267E96),
QWord($BA7C9045F12C7F99),QWord($24A19947B3916CF7),QWord($0801F2E2858EFC16),
QWord($636920D871574E69),QWord($A458FEA3F4933D7E),QWord($0D95748F728EB658),
QWord($718BCD5882154AEE),QWord($7B54A41DC25A59B5),QWord($9C30D5392AF26013),
QWord($C5D1B023286085F0),QWord($CA417918B8DB38EF),QWord($8E79DCB0603A180E),
QWord($6C9E0E8BB01E8A3E),QWord($D71577C1BD314B27),QWord($78AF2FDA55605C60),
QWord($E65525F3AA55AB94),QWord($5748986263E81440),QWord($55CA396A2AAB10B6),
QWord($B4CC5C341141E8CE),QWord($A15486AF7C72E993),QWord($B3EE1411636FBC2A),
QWord($2BA9C55D741831F6),QWord($CE5C3E169B87931E),QWord($AFD6BA336C24CF5C),
QWord($7A32538128958677),QWord($3B8F48986B4BB9AF),QWord($C4BFE81B66282193),
QWord($61D809CCFB21A991),QWord($487CAC605DEC8032),QWord($EF845D5DE98575B1),
QWord($DC262302EB651B88),QWord($23893E81D396ACC5),QWord($0F6D6FF383F44239),
QWord($2E0B4482A4842004),QWord($69C8F04A9E1F9B5E),QWord($21C66842F6E96C9A),
QWord($670C9C61ABD388F0),QWord($6A51A0D2D8542F68),QWord($960FA728AB5133A3),
QWord($6EEF0B6C137A3BE4),QWord($BA3BF0507EFB2A98),QWord($A1F1651D39AF0176),
QWord($66CA593E82430E88),QWord($8CEE8619456F9FB4),QWord($7D84A5C33B8B5EBE),
QWord($E06F75D885C12073),QWord($401A449F56C16AA6),QWord($4ED3AA62363F7706),
QWord($1BFEDF72429B023D),QWord($37D0D724D00A1248),QWord($DB0FEAD349F1C09B),
QWord($075372C980991B7B),QWord($25D479D8F6E8DEF7),QWord($E3FE501AB6794C3B)

);

ORIG_S : TBlowFishSArray = (
(
QWord($976CE0BD04C006BA),QWord($C1A94FB6409F60C4),QWord($5E5C9EC2196A2463),
QWord($68FB6FAF3E6C53B5),QWord($1339B2EB3B52EC6F),QWord($6DFC511F9B30952C),
QWord($CC814544AF5EBD09),QWord($BEE3D004DE334AFD),QWord($660F2807192E4BB3),
QWord($C0CBA85745C8740F),QWord($D20B5F39B9D3FBDB),QWord($5579C0BD1A60320A),
QWord($D6A100C6402C7279),QWord($679F25FEFB1FA3CC),QWord($8EA5E9F8DB3222F8),
QWord($3C7516DFFD616B15),QWord($2F501EC8AD0552AB),QWord($323DB5FAFD238760),
QWord($53317B483E00DF82),QWord($9E5C57BBCA6F8CA0),QWord($1A87562EDF1769DB),
QWord($D542A8F6287EFFC3),QWord($AC6732C68C4F5573),QWord($695B27B0BBCA58C8),
QWord($E1FFA35DB8F011A0),QWord($10FA3D98FD2183B8),QWord($4AFCB56C2DD1D35B),
QWord($9A53E479B6F84565),QWord($D28E49BC4BFB9790),QWord($E1DDF2DAA4CB7E33),
QWord($62FB1341CEE4C6E8),QWord($EF20CADA36774C01),QWord($D07E9EFE2BF11FB4),
QWord($95DBDA4DAE909198),QWord($EAAD8E716B93D5A0),QWord($D08ED1D0AFC725E0),
QWord($8E3C5B2F8E7594B7),QWord($8FF6E2FBF2122B64),QWord($8888B812900DF01C),
QWord($4FAD5EA0688FC31C),QWord($D1CFF191B3A8C1AD),QWord($2F2F2218BE0E1777),
QWord($EA752DFE8B021FA1),QWord($E5A0CC0FB56F74E8),QWord($18ACF3D6CE89E299),
QWord($B4A84FE0FD13E0B7),QWord($7CC43B81D2ADA8D9),QWord($165FA26680957705),
QWord($93CC7314211A1477),QWord($E6AD206577B5FA86),QWord($C75442F5FB9D35CF),
QWord($EBCDAF0C7B3E89A0),QWord($D6411BD3AE1E7E49),QWord($00250E2D2071B35E),
QWord($226800BB57B8E0AF),QWord($2464369BF009B91E),QWord($5563911D59DFA6AA),
QWord($78C14389D95A537F),QWord($207D5BA202E5B9C5),QWord($832603766295CFA9),
QWord($11C819684E734A41),QWord($B3472DCA7B14A94A),QWord($1B5100529A532915),
QWord($D60F573FBC9BC6E4),QWord($2B60A47681E67400),QWord($08BA6FB5571BE91F),
QWord($F296EC6B2A0DD915),QWord($B6636521E7B9F9B6),QWord($FF34052EC5855664),
QWord($53B02D5DA99F8FA1),QWord($08BA47996E85076A),QWord($4B7A70E9B5B32944),
QWord($DB75092EC4192623),QWord($AD6EA6B049A7DF7D),QWord($9CEE60B88FEDB266),
QWord($ECAA8C71699A18FF),QWord($5664526CC2B19EE1),QWord($193602A575094C29),
QWord($A0591340E4183A3E),QWord($3F54989A5B429D65),QWord($6B8FE4D699F73FD6),
QWord($A1D29C07EFE830F5),QWord($4D2D38E6F0255DC1),QWord($4CDD20868470EB26),
QWord($6382E9C6021ECC5E),QWord($09686B3F3EBAEFC9),QWord($3C9718146B6A70A1),
QWord($687F358452A0E286),QWord($B79C5305AA500737),QWord($3E07841C7FDEAE5C),
QWord($8E7D44EC5716F2B8),QWord($B03ADA37F0500C0D),QWord($F01C1F040200B3FF),
QWord($AE0CF51A3CB574B2),QWord($25837A58DC0921BD),QWord($D19113F97CA92FF6),
QWord($9432477322F54701),QWord($3AE5E58137C2DADC),QWord($C8B576349AF3DDA7),
QWord($A94461460FD0030E),QWord($ECC8C73EA4751E41),QWord($E238CD993BEA0E2F),
QWord($3280BBA1183EB331),QWord($4E548B384F6DB908),QWord($6F420D03F60A04BF),
QWord($2CB8129024977C79),QWord($5679B072BCAF89AF),QWord($DE9A771FD9930810),
QWord($B38BAE12DCCF3F2E),QWord($5512721F2E6B7124),QWord($501ADDE69F84CD87),
QWord($7A5847187408DA17),QWord($BC9F9ABCE94B7D8C),QWord($EC7AEC3ADB851DFA),
QWord($63094366C464C3D2),QWord($EF1C18473215D808),QWord($DD433B3724C2BA16),
QWord($12A14D432A65C451),QWord($50940002133AE4DD),QWord($71DFF89E10314E55),
QWord($81AC77D65F11199B),QWord($043556F1D7A3C76B),QWord($3C11183B5924A509),
QWord($F28FE6ED97F1FBFA),QWord($9EBABF2C1E153C6E),QWord($86E34570EAE96FB1),
QWord($860E5E0A5A3E2AB3),QWord($771FE71C4E3D06FA),QWord($2965DCB999E71D0F),
QWord($803E89D65266C825),QWord($2E4CC9789C10B36A),QWord($C6150EBA94E2EA78),
QWord($A6FC3C531E0A2DF4),QWord($F2F74EA7361D2B3D),QWord($1939260F19C27960),
QWord($5223A708F71312B6),QWord($EBADFE6EEAC31F66),QWord($E3BC4595A67BC883),
QWord($B17F37D1018CFF28),QWord($C332DDEFBE6C5AA5),QWord($6558218568AB9702),
QWord($EECEA50FDB2F953B),QWord($2AEF7DAD5B6E2F84),QWord($1521B62829076170),
QWord($ECDD4775619F1510),QWord($13CCA830EB61BD96),QWord($0334FE1EAA0363CF),
QWord($B5735C904C70A239),QWord($D59E9E0BCBAADE14),QWord($EECC86BC60622CA7),
QWord($9CAB5CABB2F3846E),QWord($648B1EAF19BDF0CA),QWord($A02369B9655ABB50),
QWord($40685A323C2AB4B3),QWord($319EE9D5C021B8F7),QWord($9B540B19875FA099),
QWord($95F7997E623D7DA8),QWord($F837889A97E32D77),QWord($11ED935F16681281),
QWord($0E358829C7E61FD6),QWord($96DEDFA17858BA99),QWord($57F584A51B227263),
QWord($9B83C3FF1AC24696),QWord($CDB30AEB532E3054),QWord($8FD948E46DBC3128),
QWord($58EBF2EF34C6FFEA),QWord($FE28ED61EE7C3C73),QWord($5D4A14D9E864B7E3),
QWord($42105D14203E13E0),QWord($45EEE2B6A3AAABEA),QWord($DB6C4F15FACB4FD0),
QWord($C742F442EF6ABBB5),QWord($654F3B1D41CD2105),QWord($D81E799E86854DC7),
QWord($E44B476A3D816250),QWord($CF62A1F25B8D2646),QWord($FC8883A0C1C7B6A3),
QWord($7F1524C369CB7492),QWord($47848A0B5692B285),QWord($095BBF00AD19489D),
QWord($1462B17423820D00),QWord($58428D2A0C55F5EA),QWord($1DADF43E233F7061),
QWord($3372F0928D937E41),QWord($D65FECF16C223BDB),QWord($7CDE3759CBEE7460),
QWord($4085F2A7CE77326E),QWord($A607808419F8509E),QWord($E8EFD85561D99735),
QWord($A969A7AAC50C06C2),QWord($5A04ABFC800BCADC),QWord($9E447A2EC3453484),
QWord($FDD567050E1E9EC9),QWord($DB73DBD3105588CD),QWord($675FDA79E3674340),
QWord($C5C43465713E38D8),QWord($3D28F89EF16DFF20),QWord($153E21E78FB03D4A),
QWord($E6E39F2BDB83ADF7),QWord($E93D5A68948140F7),QWord($F64C261C94692934),
QWord($411520F77602D4F7),QWord($BCF46B2ED4A10068),QWord($D40824713320F46A),
QWord($43B7D4B7500061AF),QWord($1E39F62E97244546),QWord($14214F74BF8B8840),
QWord($4D95FC1D96B591AF),QWord($70F4DDD366A02F45),QWord($BFBC09EC03BD9785),
QWord($7FAC6DD031CB8504),QWord($96EB27B355FD3941),QWord($DA2547E6ABCA0A9A),
QWord($28507825530429F4),QWord($0A2C86DAE9B66DFB),QWord($68DC1462D7486900),
QWord($680EC0A427A18DEE),QWord($4F3FFEA2E887AD8C),QWord($B58CE0067AF4D6B6),
QWord($AACE1E7CD3375FEC),QWord($CE78A399406B2A42),QWord($20FE9E35D9F385B9),
QWord($EE39D7AB3B124E8B),QWord($1DC9FAF74B6D1856),QWord($26A36631EAE397B2),
QWord($3A6EFA74DD5B4332),QWord($6841E7F7CA7820FB),QWord($FB0AF54ED8FEB397),
QWord($454056ACBA489527),QWord($55533A3A20838D87),QWord($FE6BA9B7D096954B),
QWord($55A867BCA1159A58),QWord($CCA9296399E1DB33),QWord($A62A4A563F3125F9),
QWord($5EF47E1C9029317C),QWord($FDF8E80204272F70),QWord($80BB155C05282CE3),
QWord($95C11548E4C66D22),QWord($48C1133FC70F86DC),QWord($07F9C9EE41041F0F),
QWord($404779A45D886E17),QWord($325F51EBD59BC0D1),QWord($F2BCC18F41113564),
QWord($257B7834602A9C60),QWord($DFF8E8A31F636C1B),QWord($0E12B4C202E1329E),
QWord($AF664FD1CAD18115),QWord($6B2395E0333E92E1),QWord($3B240B62EEBEB922),
QWord($85B2A20EE6BA0D99),QWord($DE720C8C2DA2F728),QWord($D012784595B794FD),
QWord($647D0862E7CCF5F0),QWord($5449A36F877D48FA),QWord($C39DFD27F33E8D1E),
QWord($0A476341992EFF74)),

(
QWord($3A6F6EABF4F8FD37),QWord($A812DC60A1EBDDF8),QWord($991BE14CDB6E6B0D),
QWord($C67B55106D672C37),QWord($2765D43BDCD0E804),QWord($F1290DC7CC00FFA3),
QWord($B5390F92690FED0B),QWord($667BA0FBCEDB7D9C),QWord($A091CF0BD9155EA3),
QWord($BB132F88515BAD24),QWord($7B9479BF763BD6EB),QWord($37392EB3CC115979),
QWord($8026E297F42E312D),QWord($6842ADA7C66A2B3B),QWord($12754CCC782EF11C),
QWord($6A124237B79251E7),QWord($06A1BBE64BFB6350),QWord($1A6B101811CAEDFA),
QWord($3D25BDD8E2E1C3C9),QWord($444216590A121386),QWord($D90CEC6ED5ABEA2A),
QWord($64AF674EDA86A85F),QWord($BEBFE98864E4C3FE),QWord($9DBC8057F0F7C086),
QWord($60787BF86003604D),QWord($D1FD8346F6381FB0),QWord($7745AE04D736FCCC),
QWord($83426B33F01EAB71),QWord($B08041873B005E5F),QWord($77A057BEBDE8AE24),
QWord($55464299BF582E61),QWord($4E58F48FF2DDFDA2),QWord($F474EF388789BDC2),
QWord($5366F9C3C8B38E74),QWord($B475F25546FCD9B9),QWord($7AEB26618B1DDF84),
QWord($846A0E79915F95E2),QWord($466E598E20B45770),QWord($8CD55591C802DE4C),
QWord($B90BACE1BB8205D0),QWord($11A862487574A99E),QWord($B77F19B6E0A9DC09),
QWord($662D09A1C4324633),QWord($E85A1F0209F0BE8C),QWord($4A99A0251D6EFE10),
QWord($1AB93D1D0BA5A4DF),QWord($A186F20F2868F169),QWord($DCB7DA83573906FE),
QWord($A1E2CE9B4FCD7F52),QWord($50115E01A70683FA),QWord($A002B5C40DE6D027),
QWord($9AF88C27773F8641),QWord($C3604C0661A806B5),QWord($F0177A28C0F586E0),
QWord($006058AA30DC7D62),QWord($11E69ED72338EA63),QWord($53C2DD94C2C21634),
QWord($BBCBEE5690BCB6DE),QWord($EBFC7DA1CE591D76),QWord($6F05E4094B7C0188),
QWord($39720A3D7C927C24),QWord($86E3725F724D9DB9),QWord($1AC15BB4D39EB8FC),
QWord($ED54557808FCA5B5),QWord($D83D7CD34DAD0FC4),QWord($1E50EF5EB161E6F8),
QWord($A28514D96C51133C),QWord($6FD5C7E756E14EC4),QWord($362ABFCEDDC6C837),
QWord($D79A323492638212),QWord($670EFA8E406000E0),QWord($3A39CE37D3FBF5CF),
QWord($ABC277375AC52D1B),QWord($5CB0679E4FA33742),QWord($D382274099BC9BBE),
QWord($D5118E9DBF0F7315),QWord($D62D1C7EC700C47B),QWord($B78C1B6B21A19045),
QWord($B26EB1BE6A366EB4),QWord($5748AB2FBC946E79),QWord($C6A376D26549C2C8),
QWord($530FF8EE468DDE7D),QWord($D5730A1D4CD04DC6),QWord($2939BBDBA9BA4650),
QWord($AC9526E8BE5EE304),QWord($A1FAD5F06A2D519A),QWord($63EF8CE29A86EE22),
QWord($C089C2B843242EF6),QWord($A51E03AA9CF2D0A4),QWord($83C061BA9BE96A4D),
QWord($8FE51550BA645BD6),QWord($2826A2F9A73A3AE1),QWord($4BA99586EF5562E9),
QWord($C72FEFD3F752F7DA),QWord($3F046F6977FA0A59),QWord($80E4A91587B08601),
QWord($9B09E6AD3B3EE593),QWord($E990FD5A9E34D797),QWord($2CF0B7D9022B8B51),
QWord($96D5AC3A017DA67D),QWord($D1CF3ED67C7D2D28),QWord($1F9F25CFADF2B89B),
QWord($5AD6B4725A88F54C),QWord($E029AC71E019A5E6),QWord($47B0ACFDED93FA9B),
QWord($E8D3C48D283B57CC),QWord($F8D5662979132E28),QWord($785F0191ED756055),
QWord($F7960E44E3D35E8C),QWord($15056DD488F46DBA),QWord($03A161250564F0BD),
QWord($C3EB9E153C9057A2),QWord($97271AECA93A072A),QWord($1B3F6D9B1E6321F5),
QWord($F59C66FB26DCF319),QWord($7533D928B155FDF5),QWord($035634828ABA3CBB),
QWord($28517711C20AD9F8),QWord($ABCC5167CCAD925F),QWord($4DE817513830DC8E),
QWord($379D58629320F991),QWord($EA7A90C2FB3E7BCE),QWord($5121CE64774FBE32),
QWord($A8B6E37EC3293D46),QWord($48DE53696413E680),QWord($A2AE0810DD6DB224),
QWord($69852DFD09072166),QWord($B39A460A6445C0DD),QWord($586CDECF1C20C8AE),
QWord($5BBEF7DD1B588D40),QWord($CCD2017F6BB4E3BB),QWord($DDA26A7E3A5AFF45),
QWord($3E350A44BCB4CDD5),QWord($72EACEA8FA6484BB),QWord($8D6612AEBF3C6F47),
QWord($D29BE463542F5D9E),QWord($AEC2771BF64E6370),QWord($740E0D8DE75B1358),
QWord($F8721671AF537D5D),QWord($4040CB084EB4E2CC),QWord($34D2466A0115AF84),
QWord($E1B0042895983A1D),QWord($06B89FB4CE6EA048),QWord($6F3F3B823520AB82),
QWord($011A1D4B277228F8),QWord($611560B1E7933FDC),QWord($BB3A792B344525BD),
QWord($A08839E151CE794B),QWord($2F32C9B7A01FBAC9),QWord($E01CC87EBCC7D1F6),
QWord($CF0111C3A1E8AAC7),QWord($1A908749D44FBD9A),QWord($D0DADECBD50ADA37),
QWord($0339C32AC6913667),QWord($8DF9317CE0B12B4F),QWord($F79E59B743F5BB3A),
QWord($F2D519FF27D9459C),QWord($BF97222C15E6FC2A),QWord($0F91FC719B941525),
QWord($FAE59361CEB69CEB),QWord($C2A8645912BAA8D1),QWord($B6C1075EE3056A0C),
QWord($10D25065CB03A442),QWord($E0EC6E0E1698DB3B),QWord($4C98A0BE3278E964),
QWord($9F1F9532E0D392DF),QWord($D3A0342B8971F21E),QWord($1B0A74414BA3348C),
QWord($C5BE7120C37632D8),QWord($DF359F8D9B992F2E),QWord($E60B6F470FE3F11D),
QWord($E54CDA541EDAD891),QWord($CE6279CFCD3E7E6F),QWord($1618B166FD2C1D05),
QWord($848FD2C5F6FB2299),QWord($F523F357A6327623),QWord($93A8353156CCCD02),
QWord($ACF081625A75EBB5),QWord($6E16369788D273CC),QWord($DE96629281B949D0),
QWord($4C50901B71C65614),QWord($E6C6C7BD327A140A),QWord($45E1D006C3F27B9A),
QWord($C9AA53FD62A80F00),QWord($BB25BFE235BDD2F6),QWord($71126905B2040222),
QWord($B6CBCF7CCD769C2B),QWord($53113EC01640E3D3),QWord($38ABBD602547ADF0),
QWord($BA38209CF746CE76),QWord($77AFA1C520756060),QWord($85CBFE4E8AE88DD8),
QWord($7AAAF9B04CF9AA7E),QWord($1948C25C03FB8A8C),QWord($01C36AE4D6EBE2F9),
QWord($90D4F869A65CDEA0),QWord($3F09252DC208E69F),QWord($B74E6132CE77E25B),
QWord($578FDFE33AC372E6),QWord($B83ACB022002397A),QWord($6EC6FB5BFFCFD4DD),
QWord($4CBF5ED1F43FE582),QWord($3EF4E8232D152AF0),QWord($E718C97059BD9820),
QWord($1F4A9D62E7A529BA),QWord($89E1248D3BF88656),QWord($C5114D0EBC4CEE16),
QWord($034D8A3920E47882),QWord($E9AE8FBDE3ABDC1F),QWord($6DA51E525DB2BAE1),
QWord($01F86E7A6D9C68A9),QWord($2708FCD9293CBC0C),QWord($B03C87F8A8AD2C2F),
QWord($00424EEBCACB452D),QWord($89CC71FCD59C7F91),QWord($7F0622BC6D8A08B1),
QWord($834D21326884CA82),QWord($E3AACBF37786F2FA),QWord($2CAB6E3DCE535AD1),
QWord($F20AC607C6B8E14F),QWord($5EB4388E775014A6),QWord($656665F7B64A43E4),
QWord($BA383D01B2E41079),QWord($8EB2986F909E0CA4),QWord($1F7B37772C126030),
QWord($85088718C4E7D1BD),QWord($4065FFCE8392FD8A),QWord($AA36D12BB4C8C9D0),
QWord($994FB0B714F96818),QWord($F9A53998A0A178C6),QWord($2684A81E8AE972F6),
QWord($B8425EB67A29D486),QWord($551BD719AF32C189),QWord($D5145505DC81D53E),
QWord($48424EDAB796EF46),QWord($A0498F03667DEEDE),QWord($03AC0AB3C497733D),
QWord($5316A89130A88FCC),QWord($9604440ACEEB893A),QWord($7725B82B0E1EF69D),
QWord($302A5C8EE7B84DEF),QWord($5A31B096C9EBF88D),QWord($512D788E7E4002EE),
QWord($87E02AF6C358A1BB),QWord($02E8D7AFDF9FB0E7),QWord($790E942A3B3C1ABA),
QWord($C6FFA7AF9DF796F9),QWord($321BB9940174A8A8),QWord($ED22162CCFF1BB99),
QWord($DAA8D551A4D5E44B),QWord($ECDDE3ECA80DC509),QWord($0393EEF272523D31),
QWord($D48E3A1C224EB65E),QWord($6052C3A42109C32F),QWord($052EE388ED9F7EA9),
QWord($91C62F9777B55BA0)),

(
QWord($150CBCA33AEC6525),QWord($DF31838343A9CE26),QWord($9362AD8B0134140B),
QWord($8DF5CF811E9FF559),QWord($167F05643812F4E0),QWord($588A52B0CBB8E944),
QWord($EF5B16A373C4EDA1),QWord($7DFCFEEAF54BCBBE),QWord($8773E3D2C531DCD0),
QWord($55C4672952774F3A),QWord($57CA6BC0467D3A3B),QWord($24778425B7991E9A),
QWord($DD825C26E452C8EE),QWord($FCACDE1E84833AF3),QWord($61211D031732C131),
QWord($CCADB247E606BE8C),QWord($712B39F188B4EF39),QWord($3A9FCDC5C5755169),
QWord($1FF6994F39829CB0),QWord($110165733343CBEB),QWord($61D3D0B444F30AEF),
QWord($A8AE73752A3A1C9D),QWord($B4B70914D6AB250C),QWord($853B7328495F948F),
QWord($D2A4ED8E6CF751E4),QWord($C320BB75D9CAA0B3),QWord($8BA562624E84B03F),
QWord($EEA8076E74A07FE5),QWord($8039E00C36FFDAF8),QWord($03731358B9E671B9),
QWord($DAC4CE1CB25B10ED),QWord($4DD3D5B1FCF2B480),QWord($4634F57925EAC400),
QWord($A9AC55EA728932DF),QWord($06041D055D31F502),QWord($C539C2E32B89D9DB),
QWord($5BCC0A98C05BFD6F),QWord($1B2506222E21BE0E),QWord($60973B04ECD54A67),
QWord($B54FE638A6ED6615),QWord($981A910A5D92928D),QWord($AC6FC697E73C63AD),
QWord($456EDF5F457A8145),QWord($51875A64CD3099F1),QWord($69B5F18A8C73EE0B),
QWord($5E57368F6C7AF4BB),QWord($7A595926AAB49EC6),QWord($8AC8FCFB8016CBDB),
QWord($8BBC1F476982C711),QWord($85C7DA7A58811477),QWord($CD67FAD1D764D9B4),
QWord($C81029505CD09DA5),QWord($1BB1F14795167D80),QWord($0366046DAF1DACA1),
QWord($A2247B2311301A54),QWord($791D99C67A4FB7CF),QWord($277449A409E57492),
QWord($35C9A57E5E7F500A),QWord($B9A62A8AD5242A6B),QWord($A13378599CDA3346),
QWord($148740474328BA08),QWord($EB81D51F3248896A),QWord($8007D85D0F6E8DDA),
QWord($8250BDAFCE2EE042),QWord($897EE0225F003612),QWord($3BA18F9026314076),
QWord($7824035A3B57E2D5),QWord($8E78AED1E90DC600),QWord($90C15EA624609519),
QWord($B72EC3F8663DDCEF),QWord($EF574232194195B6),QWord($20C9203FF993700E),
QWord($C1A44A7CBAEDE941),QWord($F98AD4C4F8F43F26),QWord($F060DD080EB9DF1D),
QWord($9B33618855EEB734),QWord($6C56D0D433A0D6E4),QWord($182885B21DAFB611),
QWord($F04D46F750CDCB91),QWord($407B4733D7A2B343),QWord($0269AC52B520773C),
QWord($8C910B9F7680E5F6),QWord($C79AD874FD97CFE6),QWord($DEB78FB3279EE2A1),
QWord($7428670533645AA5),QWord($554438084F1FC8DC),QWord($116CC402D4F14617),
QWord($6183CD9EE11D5B9E),QWord($195D2523EF2F8E07),QWord($D836767C44811938),
QWord($AD366C9FDAB0065B),QWord($2C5BD6F663F7D10E),QWord($79104BD1D7C3E497),
QWord($432DD056932F8F20),QWord($878025A708E90347),QWord($CDAC2120753A3201),
QWord($9C6BE2FD03F7B32B),QWord($0AB85150D75A8F1E),QWord($8FBF70ECE0C67DF9),
QWord($71F250E41ABB9BF0),QWord($3F17F108EDBFDC76),QWord($37DCD54A48CCE570),
QWord($8011BB358BC70BFB),QWord($8194F91BD0816B10),QWord($5B3FE6415C6C5D42),
QWord($39D03AD66881CC3B),QWord($95ED2BBA9B2DF677),QWord($7DB8057E0E1A306A),
QWord($9C73B22B0F6FAE4C),QWord($1477324E85041C8A),QWord($31F1B31CB03F5338),
QWord($3A3E70F003BDA3D7),QWord($21507C16990FC0DC),QWord($EC48EB1B6E5AA777),
QWord($9DC7E5750C9490FC),QWord($E8B70BB69C2462DE),QWord($58F9D6E286C90D9E),
QWord($393954BDB5F2A4FD),QWord($A1CAD7C66DEDFC47),QWord($4664D28514E21038),
QWord($A34AE5A60CC88295),QWord($26D74884885A7DAA),QWord($AB0E25AAE328DACF),
QWord($3D75CA5C9315B071),QWord($A8F86A7A869A2A3B),QWord($73C244966E32834B),
QWord($247DEE3AB3A3E118),QWord($0E3E864B85639D62),QWord($88B3363FB3A767E5),
QWord($4B847B5E39540816),QWord($7538DE9741B0D6D8),QWord($ED77D7107636B96C),
QWord($F57D4C5B2066FFB3),QWord($17877F49FF0478D2),QWord($759AF12C0E64DCF8),
QWord($547B59F127BCEC3D),QWord($18CE85B88418FE15),QWord($F07CF6B4A5827EC1),
QWord($ED2893DE8A5FE68B),QWord($4E112D7347572CCB),QWord($AF86678002E62D57),
QWord($E2994ED7B1C7EBA4),QWord($01AAD3603470F655),QWord($8F1BB1A1E9FC7E99),
QWord($995672B0A2FA5702),QWord($15B4DE2A4F23088C),QWord($F2F97EC938568716),
QWord($61A4FFCCF419CF11),QWord($CCFACBCEBBCA28E3),QWord($784DD43E1150DEE3),
QWord($BC060BA08223C3D9),QWord($4BDA38C78D9A7575),QWord($4C007EC0AB58EFBB),
QWord($5CC287D0F064EA5C),QWord($85C0368AAF2AD213),QWord($0EAAC3CDCE740241),
QWord($FB003622BFAA4ABB),QWord($583B589762A96A1A),QWord($D06C7F57FF53F144),
QWord($DD67F044D1C28282),QWord($8BDA0388548D04D4),QWord($4E4DAB2DB00734EE),
QWord($D46E84086EFCE2BD),QWord($39C9FB1B50898815),QWord($C67E79C49FDEDF05),
QWord($10DA0B2AF8B32839),QWord($06C91AF07F24F3BC),QWord($FB10760CB49F0850),
QWord($D2CC97D8EC63395B),QWord($9DB31D42A7CEC064),QWord($F90E398D479ACAB7),
QWord($07BE140A28F3754D),QWord($728C72A3F53F82D5),QWord($BDEFD866BF22F70B),
QWord($655E9A03A6F11D63),QWord($4A7132F693E833A5),QWord($5E9274B4015F4E35),
QWord($6A59DCDD9D1E85EF),QWord($137D430AA6BE0B0F),QWord($12A13EA30191692D),
QWord($91A1E7691CC2852D),QWord($AEE7300FDAF3860E),QWord($408AC32A6A50C876),
QWord($996E523B8C04F81B),QWord($417F8595AA4CC582),QWord($FCB3B8E23C2156FC),
QWord($CD71AECD32A95C70),QWord($7924389BD1047541),QWord($9516C77F3DF7F5A6),
QWord($B1384D6FCB22F10B),QWord($BC73CA3B1EC4F4F4),QWord($B9DEE7FB93A467F7),
QWord($D638221089A1405C),QWord($FD2FB6A54BC618B6),QWord($96F3BB267289FE18),
QWord($EB855796F09854D1),QWord($DC9E8AFCAF106FDB),QWord($F5A238571B1815CA),
QWord($0A6F7CB51513353F),QWord($8AC43941D1600967),QWord($BA7A830ACD6CE82C),
QWord($29AB1461C4BE5ECF),QWord($BF756728EE1651D2),QWord($8351FEF2D9A56503),
QWord($1160D3CCB0767848),QWord($82FA32727DC7523B),QWord($409D85E9F46167C9),
QWord($EF335BFEDE10E539),QWord($360295CD5E0F347A),QWord($9DC0DA0142AC93FC),
QWord($32C3BEC9171678B9),QWord($4C78A82B7C2C748F),QWord($3828095E064D62C9),
QWord($F49CB2BE9C9F0126),QWord($321485CB4773E463),QWord($0E24D3491B7D32E8),
QWord($5D15EE76A962F764),QWord($DEB15D4574DB8D32),QWord($099610F3B3BD25A7),
QWord($8A4A1E30E3A7F974),QWord($11B6004CDA44BF5E),QWord($9848711F5104ADE1),
QWord($4A3B2AC8CF048420),QWord($2F073250BE5F1DCC),QWord($59F58B34F5410B4B),
QWord($172EDC27C8B798C9),QWord($E7FBA7BEE8131C63),QWord($3DA9D7944FA593B6),
QWord($684A0C6BEA248658),QWord($6A33F10A3A76F7C9),QWord($490BCC010090EDAC),
QWord($643FE36AA33A5490),QWord($A3D5F8B48CF3FCE6),QWord($F53EF83439DC6EED),
QWord($CB7A2AD77357022B),QWord($BDA4E4538DB6C313),QWord($88DF6453A5E9EA24),
QWord($5DC52D198ABE6869),QWord($6BDB8D96E21B92AB),QWord($8C3DB835FFCEDFCA),
QWord($AE947E3B49A2B4BD),QWord($298CCC36A3D7193B),QWord($922D5A7ED0C5161D),
QWord($E4CD940BE06728C4),QWord($EF5CB80D23E73708),QWord($29C4FD6F93F61230),
QWord($4E30CF0507C74C09),QWord($C04C1B3D943D5EC6),QWord($03E1291945FDD157),
QWord($11348FB5CB36DBA1),QWord($C5FE692324A33B0E),QWord($499425F2FA87319C),
QWord($587D2E1DE3BF7A76),QWord($3D1A1D39600D94E4),QWord($88019070C7BAEDFD),
QWord($13733CC160299767)),

(
QWord($E3EDE1160BC75684),QWord($44ABBD60A47D8E20),QWord($C589A0633522CB00),
QWord($E87815164ACE93CC),QWord($6B9E5D20A12CFE3D),QWord($11371EEB584A4436),
QWord($A6DF8D4BB44831C6),QWord($852C8C2A711EB439),QWord($6DB68C03A9FB89F3),
QWord($FC170E64EA76B5B0),QWord($8DBE0E5F805CAD36),QWord($7530AC9C6CF68E35),
QWord($891A338814ED7939),QWord($E8214B6BDB3206B1),QWord($E4A66072C4497E62),
QWord($FC14BB667F4EC9A0),QWord($5648857D9958EF6C),QWord($CE2487DDB79E1FAC),
QWord($49687EF7BFB13209),QWord($72F4723CF6652529),QWord($D291228D7308942B),
QWord($F00DE97DE596928C),QWord($F7017FC9A08BF910),QWord($1AB9C11568F592BE),
QWord($B48A4ECE91C1B9D0),QWord($0BC340B19ED82B04),QWord($435D352B0BD8FCEF),
QWord($3544FFDBB90F5CEE),QWord($ACB9B26C18141308),QWord($F0D3923E920D0C84),
QWord($67D95EEE3D2892EE),QWord($CBBF9791DAC6ABFA),QWord($2107CA17EEB2D41F),
QWord($0A15B8536117F7F8),QWord($E38EFD17847C5965),QWord($299F455128902084),
QWord($E3C19C842DA0FC7F),QWord($438542355030A02B),QWord($09B20BFDA067ADC3),
QWord($207E8E2296B7B8C3),QWord($AE2034D10DF42664),QWord($8AEC66374C6A4CAF),
QWord($021C5E382756DF1F),QWord($DAB397892E258D3F),QWord($BBEB27F2593CA22C),
QWord($02F538557C761FBD),QWord($5D1F93565BD28C1D),QWord($F93CE73599B8AA44),
QWord($118489EF0E0ED804),QWord($26E9A60FC55BA1A7),QWord($2D592A941EE1BBCA),
QWord($F3E4EA86BB639898),QWord($8A2C6F5594D4EF71),QWord($D3DD68B3E2E12B5D),
QWord($E3A3F757D7EE1599),QWord($A4C0E18D9997F82D),QWord($BD32A8C683CB1B29),
QWord($A059953A4F92D812),QWord($BB2289A9855D888C),QWord($677D6EF341DA321A),
QWord($2B4FFC49B1821BD2),QWord($0257E7E26F413397),QWord($CB31728801CB390C),
QWord($3CB685B21CB3FBA1),QWord($788DB8A125D301BB),QWord($63BFFA8112F19D42),
QWord($A26082BE05E4E1AC),QWord($2FA9937F2125AB76),QWord($E5E315E2266C09BD),
QWord($DC5B8B66D671F135),QWord($7F354193FABCB03A),QWord($3C9D682D7F698313),
QWord($8BA04039FF140FDC),QWord($312A0A394E8C9048),QWord($1A2290ECB9F91D01),
QWord($28CD49E7AF451EC0),QWord($0F6ADBDE5FFFC10A),QWord($C53C8C75FA7DCF99),
QWord($01FC98CCB44CA6F3),QWord($6BD7A58209C968C0),QWord($AC23BCFD1A4FA2E1),
QWord($7FC86C6FB1B0E97C),QWord($9ED901F601EAF9F0),QWord($95F1081A1BEBBF37),
QWord($DB26CD0DA39B1F05),QWord($4C6E73DF5FB3F0FA),QWord($3EB86A198B26131C),
QWord($CAC5592422935EBD),QWord($95EACD29BF617EFA),QWord($2F41A05EFFE82B79),
QWord($A86A40EF4867CAE7),QWord($9C39B9117A125986),QWord($E37D32CE39DBCB51),
QWord($3F68378EB6844A13),QWord($7CC5A27CA798F90C),QWord($0620A80357EC10A5),
QWord($EB8432E5703111DE),QWord($E859F372BE354560),QWord($82656F21A2C57145),
QWord($14A9B7E1C553EA68),QWord($D2D5DD6F9307BB8A),QWord($14B7194207B5105C),
QWord($8184D3C3A12DDC77),QWord($519F29673D8158CE),QWord($B3C32CA30F77E983),
QWord($2128E5D7E296BBAA),QWord($B3C36F108C02701B),QWord($79799E52E875F1CF),
QWord($FADFFB023A68E76C),QWord($B092A17C4F389380),QWord($26649815211577DA),
QWord($64DF730E2B87A4FD),QWord($1A5E4F14169CFCF2),QWord($E076A65F976E635C),
QWord($DCAD57936E017E84),QWord($2123F4ADCFE3E761),QWord($7689CE9C7291B34A),
QWord($3128D6249BFEF6AF),QWord($ED98745FAD9BCAA0),QWord($E48850A31635FB8B),
QWord($06FD57BD0D326219),QWord($1746DFA4E85901E7),QWord($AE9D5A4533D88A6D),
QWord($21EA70902DE52E47),QWord($711024D7735E27C4),QWord($8348E17F014190E7),
QWord($F6DF0D50C6700128),QWord($5B70ECABA6A01C98),QWord($1114AF3E1DD46FD6),
QWord($1E3C56823076FA3D),QWord($642F2EB5A7C7C625),QWord($75655A99C51920CD),
QWord($EFC5E07D1C996040),QWord($E3048644EF5FC2BA),QWord($F381235F5959D426),
QWord($B1485DBBF14BB82B),QWord($340CA2AB7E8C3151),QWord($5901BB9DD8C93107),
QWord($1693701C2F0FD38A),QWord($2265D674ECC395FB),QWord($F951CBE74CDE4AF9),
QWord($16DE85E3E0CB3310),QWord($8B16C0143A0106FF),QWord($F125B5B3CB45F407),
QWord($79E7389F5CD3A367),QWord($5A80B1E4EDB17199),QWord($436EA05A3E377949),
QWord($3A3D4ECDE00B34A0),QWord($FFA49567F668FA93),QWord($36BCAE2201E1C17C),
QWord($EAC3650F973AD8D5),QWord($430DBB8D05F92104),QWord($5641EDCE3AC26AFB),
QWord($786AFF702A2CACB4),QWord($F6228018EEB3E205),QWord($5223B6B884DA7B94),
QWord($F69CB1725661CED8),QWord($52D65674BB06447D),QWord($10D976884A4D4E2E),
QWord($85562963AFB9FD77),QWord($3EB067C62ECCB316),QWord($B715B82C4CD5704B),
QWord($FC48C9515A696AAA),QWord($91CA3A2FC6C97B48),QWord($1E1227EEEA5AAE0F),
QWord($1FCE1B1B0149C632),QWord($FDD8AFD9F6466639),QWord($645338F3EB8392C8),
QWord($3318915B53748398),QWord($BB711937AFAA09E5),QWord($5FC32FF1B3C1FE1D),
QWord($ACA39DBE6F87B608),QWord($BED2B1B1C036C554),QWord($22DAAF24CAF4F53D),
QWord($F854E55A212471D9),QWord($7D2D07779B25563B),QWord($85ED851CDC95DBE4),
QWord($6FF966162C13E934),QWord($0886BA4D85F776EC),QWord($4FC3757CCC791C7D),
QWord($67171D83FE93E855),QWord($650428C5A5D69909),QWord($1A4AF2BC95B74227),
QWord($26AE05E937CC6E58),QWord($B6EBF6A679FC9DBC),QWord($6A289F831DB090A4),
QWord($FED906D8873153D8),QWord($8D17068C2FCB255D),QWord($9CE6F36322E5F0E9),
QWord($99505EC4777D1D28),QWord($CEA31A15FC3F4432),QWord($40EF079C4F26C32B),
QWord($F9C2CB89F0B40D03),QWord($4F10DBC4BC83DE6D),QWord($E34AE5CB8798EC34),
QWord($632D8651CC6CAF6B),QWord($CE4797BD98CA8826),QWord($C28D82EDC7BCD3B2),
QWord($2AFB56FFB2280BF9),QWord($0FAA53BA70BB13F1),QWord($B88C9BEE75F865DB),
QWord($65B9747600C27A47),QWord($7147367E52B391D3),QWord($9A10C6322889967A),
QWord($D667521FE7E68D9F),QWord($41B14D505D608D8A),QWord($2A8D5EE472197B7A),
QWord($9E0D40E0B7D84D86),QWord($8FAEF64A9F40B830),QWord($085A45BF37C66777),
QWord($DD26FE46494978CD),QWord($C336AC136D0EF2D0),QWord($93D63D524059298F),
QWord($758E39DEB59FB07A),QWord($9F5087D97A5ABFA6),QWord($EA89450159E9B256),
QWord($0B81091656CF23AC),QWord($8BDD7C9476FA3428),QWord($0CD33C02749D58A8),
QWord($C14D4DCE4D984D74),QWord($BCEA2DF9BA5EE882),QWord($FDFB1D43FA94D0CA),
QWord($EC038FBE3F78E4BC),QWord($0C83727C60A40C3B),QWord($B60283FD1A1F7DF8),
QWord($83647DDF5A6EA43A),QWord($45E2C5FA1164677D),QWord($12C2242292D83289),
QWord($67C64BB1516CF2FD),QWord($DEFC03FA52D573DD),QWord($F2DA185F6FCEF553),
QWord($280C9FAE8BEBE1C3),QWord($5C4648C6AC8E2156),QWord($39F3ACDDAAFF23EC),
QWord($426570EEFF59D494),QWord($19C3BE0BD6F859FB),QWord($23CE10871775DA1A),
QWord($BA648A66CCC808C4),QWord($51BE41EA51A28FD8),QWord($34DF5E938481F74F),
QWord($6100816D3340A466),QWord($70548249C75E5120),QWord($BA79702279A00748),
QWord($6F067133E39C6F68),QWord($D4A56F6DE85BF670),QWord($9BC03ABAA6FC1A71),
QWord($7896217952937711),QWord($04CF9FD6C613F747),QWord($ADB4E5E44BD90F88),
QWord($994FD52DF174D699),QWord($30042813012C0114),QWord($F26207D02A6E2994),
QWord($D01EA8CA8B4289E4),QWord($288B2E9FB77EC8C2),QWord($062AB5EF441C44B9),
QWord($023017B24D33E3F7)),

(
QWord($C9C549126B629FC5),QWord($13D2028D708FFC38),QWord($99BF4A68002FA9F9),
QWord($959B2BDBBF79C584),QWord($2A6AE3A9AF0DEF7F),QWord($3961DFF94E0F2FC3),
QWord($375602F6CB38090E),QWord($FF27FA5FB4361DAD),QWord($27F105C6FF7A86A7),
QWord($89C6A294797B99FD),QWord($859C810D3E650654),QWord($CD5FAEBDAA91E7A4),
QWord($80B70152513A0E3B),QWord($F3D5225D6212305A),QWord($0B2E2D04D6E0A998),
QWord($BC00734CB44E2BC4),QWord($DF6923BF27418C14),QWord($6B0D8E79BBC2202C),
QWord($4F5354ECCF70BFBD),QWord($3004547DA9FC32F9),QWord($284A7E7F49AB8971),
QWord($7C4C5273348AECA7),QWord($4AF7B4566A7D274F),QWord($41144E72C6576B5E),
QWord($94DE31E88FFF2C89),QWord($C4B22594D0138A97),QWord($63EFE7A453EA6DDF),
QWord($B611ECB4722FE373),QWord($F819C4B88FC3DF60),QWord($93E879D9871E50B2),
QWord($0AB6063FF823AB24),QWord($1D753E36002FE0A2),QWord($59155829A234E540),
QWord($2A3A5EF606E16C73),QWord($398EED502BA793DF),QWord($A2E4CD3D50E7C902),
QWord($455226EE9CC01966),QWord($E783C5FAA4A03BB8),QWord($E627076A6A926733),
QWord($7ECFF39F7A1F3091),QWord($0CFB8FC622B3ED44),QWord($3BF287327CC97604),
QWord($16588EBC87273BFD),QWord($23AD2E3D79DC9FA4),QWord($6F33F954DC3DDFAA),
QWord($8855135E2935EF4E),QWord($980A08E4CADE1B0B),QWord($0AE829C4B9832D25),
QWord($7E26508CB0E1D951),QWord($014E05A1EFEF032A),QWord($7D73E257F5E46436),
QWord($3EDDB5EABF92518C),QWord($BEB2E77DE87FBEA2),QWord($E774B1C87370FA54),
QWord($8212C2D3FD68BCD3),QWord($80FDFEBF5965C349),QWord($36E0BE298259B3C2),
QWord($7900B71C3BBDE591),QWord($4BE48027CBABC230),QWord($9AB8623D10F1DDB8),
QWord($67353FE74B23E082),QWord($14D19143CD01BBD9),QWord($9B8F27FEE6873317),
QWord($A80F9F28BE002243),QWord($95E3326A1A5D0985),QWord($E9CDFC8A83BB7D8B),
QWord($DD5BDCBFB6238048),QWord($AC5DB086974C4A25),QWord($3138A84A6EE55DF9),
QWord($B6AAFF87D1AAAB74),QWord($7D882D01EAC51639),QWord($B1D786708070B049),
QWord($2B2D6ACBBE60CA03),QWord($7E9FB44B03BB7CDD),QWord($D46CD6E2B7355F16),
QWord($28189952E1F24EF1),QWord($AB1BB900C9ED265C),QWord($E2D04CF2396E9E95),
QWord($E90B7C2BC71EDFBF),QWord($2BFD0239BCAA1AE2),QWord($D5A67AC75C09E06C),
QWord($1042C9914BD7A768),QWord($B95D116A50577DF5),QWord($4D5BA7FD9302A958),
QWord($10EFB35D7C44D04C),QWord($2840CD3BBFE8A000),QWord($6E1D0B91BF83166F),
QWord($C94BEADCE2A22661),QWord($29F3DEC66D94A52B),QWord($6DE8D82598C80D5A),
QWord($DBA35D1E9FC9976E),QWord($70C6F78CCB1DC1AC),QWord($5F89D50A1A0F5014),
QWord($89BA6ABE2675A341),QWord($2E91CA2D639705C6),QWord($0A1E10F5ECB1CDF8),
QWord($C6F4D4C76CA28E32),QWord($6C1F1960E1FFBA3D),QWord($FDABA7B749417CCB),
QWord($2EBE5734F41BB7B7),QWord($FF1B3883F1C9AB13),QWord($D2BB196566A38EFA),
QWord($7B512382658E0EA5),QWord($935BB5D4A12A9BF7),QWord($960FBAA2555C0B70),
QWord($3E525A53AED652B2),QWord($B391181888F09E91),QWord($EB9C20C7852CB5B6),
QWord($FE412D512E5FC6F7),QWord($FCE8C877AEBF0456),QWord($64E554384FAE86D3),
QWord($3F7FCB2D2961457A),QWord($EAAA1911B7B127B4),QWord($FB0705A22D459DC0),
QWord($E1BDFA1C0BFDEE98),QWord($F87B87A9F83C326A),QWord($5EF4FBCDA686CCD5),
QWord($3A6062109261F2CB),QWord($6DE5ADF39C432638),QWord($D3E5B2421AA14DF5),
QWord($B8C3E2A48D04B5AB),QWord($353AD3256B300551),QWord($2C293E7B5CD65ED1),
QWord($D1424A125365A941),QWord($2D7DABF9A99A9EF0),QWord($A22E73F4144C14B0),
QWord($B28D10B4DAEDEACE),QWord($ED1C674702891AF1),QWord($97086B1018A4CF83),
QWord($E2BFABED37127735),QWord($02F6EA6EE53CEE62),QWord($ABD1EE13E70FB008),
QWord($C74EF42B9A66ADC1),QWord($B37D68BD31CF57BA),QWord($50FE1D7E33412595),
QWord($5E1D0F55FF19FA60),QWord($60BC42EF8A205D65),QWord($F6C988FE294D1666),
QWord($58FBD06F63EB46E3),QWord($6EF46444A4CDBE82),QWord($8B5264728BD852F1),
QWord($BCAE0DB7DD7356F2),QWord($A668BEA6E240D70C),QWord($3B57508CB52C4041),
QWord($896E399798C84992),QWord($55762FF107ED577E),QWord($594A7E2FBCB2EB8F),
QWord($9AEBC80F2B38F26A),QWord($43E6DD993FE053CB),QWord($BD77A7E5122AEAAD),
QWord($E7928B16D1212579),QWord($095BD76F6E364AE7),QWord($45C5C8EC7B9A739E),
QWord($192B00027D4F1F15),QWord($C7370445FF3569DA),QWord($4D89F2C82087874D),
QWord($3001B7CAB20DC804),QWord($8E85FFCF7DE4AB5D),QWord($3BA794F55A19ADE7),
QWord($1D485BEBF50C76DA),QWord($E937C2354A69FF66),QWord($65B0FDC5E27F9505),
QWord($FDEC5B5707A41CB5),QWord($56226E58F0342F9D),QWord($E47B29225D24BE40),
QWord($0708B0582DA0D569),QWord($CAEACAFB626D88B6),QWord($D9A23DDAFC49DE6C),
QWord($0E54F8C288AE7A36),QWord($5A16100CF86D7CA0),QWord($1B400779B429DCFF),
QWord($FFDB45E8FE03B385),QWord($55883F5A18C4C190),QWord($36080E4DE983C048),
QWord($A7619E4D5D7023BA),QWord($EA1B4448FAE19DF1),QWord($920A7A62FF9BE5DA),
QWord($8D630C913A8457B3),QWord($C1317AAD19B125E4),QWord($2AC60DA85EF6A4FC),
QWord($18453D57CBE16E34),QWord($5B3D0DC184E910D3),QWord($5989E958B02A738C),
QWord($A76A21342AE3153D),QWord($6C9A6DA3A6B590C3),QWord($9294F2B261FFD9FC),
QWord($5DC7ABBC0A085E11),QWord($718F3BE7FD0598BF),QWord($8552331CA17183EC),
QWord($C702CAA9151C171F),QWord($02A6E9CD28FC2637),QWord($1D79CE9FCA06C85D),
QWord($F990DF8D4704243D),QWord($4A5C1F56FE965B3C),QWord($603347ADDC37A650),
QWord($1CAE34EB5EBB9A97),QWord($137D411F4DEFE9FE),QWord($8B1E6D8B9D92A7DD),
QWord($5888051BBD7DE708),QWord($2A05F513CD02F32D),QWord($F37A6090024DF9AA),
QWord($3E9D2A7E829CFEF2),QWord($61D25546792B1059),QWord($1B042CABCC51561C),
QWord($BA6A92F3C9C23431),QWord($DAFA0C012658242F),QWord($15388A8C97D1C6E1),
QWord($1D75A986270AE314),QWord($00AD6269F740DAF5),QWord($E11ACD2629ED5C7D),
QWord($78B8E95513F8D4C2),QWord($9B4216455EB07F9E),QWord($DE96E41B398E740F),
QWord($47F4DBAA6AF2216B),QWord($E324E52ED534622F),QWord($BFA0F89B7BF36FA7),
QWord($AD459D4846671973),QWord($BC8C0111A08779BF),QWord($BF7E8F32F914239A),
QWord($7C039FA68DCBCC5C),QWord($B2C77934F9D19EE2),QWord($2014EFF97BF6F383),
QWord($18A1E15B762D0A76),QWord($AB97671844F7D0F7),QWord($D900ED19F558726F),
QWord($C40702BEFEF81427),QWord($3C04AD1AA20E9FEC),QWord($A5994E7F4FAE1300),
QWord($C30D8C955DFFC1E9),QWord($28B67F51321D79A4),QWord($80805CDDCAAC217A),
QWord($C450000E76A4C407),QWord($E6B11477FB29BA1F),QWord($B6117B2EDB67EFDF),
QWord($90919CF5F2402660),QWord($870990DEE6953D45),QWord($59DCA2BBE1D1EFE0),
QWord($1AFC02321CB98FBD),QWord($4D64D3E14F230A06),QWord($6711320EA8A9F71A),
QWord($B30472183291CE32),QWord($50E85B556358D19B),QWord($75574349C061908A),
QWord($D8FD1AA7E4FC7A61),QWord($C01E9492FB30E932),QWord($8EE6D89FAA3D9A2E),
QWord($F8B5C1B7C0B0BA37),QWord($D81FDFDD54EB3270),QWord($9CD27406EB1FF9F7),
QWord($338CAE31F8B82E2F),QWord($85FD727AB033FA40),QWord($D702FBCBB7403203),
QWord($DE77DD60D4423BA3),QWord($4C85C9518E850016),QWord($7A70931FEE8600A5),
QWord($ADBBA0197CD52986)),

(
QWord($626263A3B381D4D1),QWord($28913CD012DA73DA),QWord($6B41DD34C1ADA2FB),
QWord($8D97DA19B167ED9B),QWord($B4812D4C5BB6CDA7),QWord($669C5EB0AB412ED2),
QWord($D3C0D38280BCADE4),QWord($961F99BE007375C9),QWord($E1CD3F0D18427E6B),
QWord($2F8E1F31AC07916B),QWord($0BCB45904214A966),QWord($080DEA4FE4402AB8),
QWord($3164DA2C4EC6D4B3),QWord($EC1AB3E0371AA403),QWord($703CC4A2D61C9EAF),
QWord($DF938277B360CC64),QWord($A809622ED1CA223A),QWord($0D0C41C61FC1CB73),
QWord($6B1D07D0F7631F64),QWord($CD6C472B23716169),QWord($7C547D101DE31920),
QWord($B8AA5D5D5A08B84D),QWord($8783FA268476EEC3),QWord($4AD24805DEA948D4),
QWord($EFD4680DE1082145),QWord($1F400A6F5CB25D7B),QWord($5830BAE8DFC1B3DD),
QWord($A52EEAC39FA9E4E6),QWord($E780295AA1FDA9CB),QWord($05292076633BD6C1),
QWord($84F452863BCCD3F0),QWord($0C0FFA13A48AC0E7),QWord($C10B071D8F38B069),
QWord($FC797F140A0C843B),QWord($55B2102F2B6F2CBA),QWord($6A65ABF4BC13F3D5),
QWord($7AB981E9ED36BC8A),QWord($01590DEDE28853A9),QWord($880A305A45927BB5),
QWord($AD60BEC2758C9434),QWord($4F5CC2D0842FFFA1),QWord($B56BC910589D9119),
QWord($29510482F6B85BCB),QWord($191C5E8C0AAC8F1D),QWord($53AEE004B3A760B2),
QWord($6C47171AE122EE50),QWord($8F882588C3AF791D),QWord($3AB1E787787F23A2),
QWord($740A3D69F574F889),QWord($563A7542EE0275A9),QWord($1D4C7F0288B0F4BD),
QWord($0B6A095505C57B60),QWord($FCDC13AC38250CC2),QWord($7C0F6DD8FEE40397),
QWord($75D040BD4F8C08B0),QWord($DA40BCD1AB461EA7),QWord($0BB2664FB361B106),
QWord($FF88C17C7ECBBDF3),QWord($CB397F5A34BD8C06),QWord($2C79097E71DC4099),
QWord($0C7E28743703E385),QWord($6B4E04AB3244D2E0),QWord($7FB8C9F38282173F),
QWord($21AFABC5CA89F8C3),QWord($3AA25BFAE9E1B88B),QWord($77E8C0B888B0B2EE),
QWord($8D7E99220B214D8B),QWord($E4C85444B026A4F2),QWord($F29A85E1881A45A5),
QWord($73105CDEBCF4939A),QWord($6B923A2FFAA6D9CD),QWord($44EB8BB4794D87DF),
QWord($0A60094B3CB6F83C),QWord($03661679A1F217E8),QWord($963C57F0455BA773),
QWord($48ED66F36494C7AA),QWord($0B20EAFD85EB9191),QWord($1C890980EFA34C0D),
QWord($BDA11E825B6095DB),QWord($3E193DB7CE3E5F19),QWord($D52A1E3477B0CC82),
QWord($489DB03838376CC8),QWord($3CEF20B0BC9F84E9),QWord($70CB14480EC62200),
QWord($246227A4A55EE063),QWord($7AF69EB03F5EA747),QWord($39FA87659196B7F3),
QWord($81ECEEFE543BB263),QWord($041009D0CF373731),QWord($62836B24EEC15E94),
QWord($B498922DBAB7E562),QWord($B8A4FCA94750926A),QWord($880C34678827F078),
QWord($D9ADAD0A8D1D0854),QWord($6F3054702D3DE155),QWord($FFF92A2DA11FD9B0),
QWord($6B012A4C96009615),QWord($95BD42D4CA1FF867),QWord($B1B1B706A1B652C4),
QWord($606C1C06836140CC),QWord($687393E9571F598B),QWord($7FB3189C01008403),
QWord($196DC6E309DD3F96),QWord($2EE2504E4FA033AF),QWord($656EDD9C1C2898DA),
QWord($1CF705B8EBE7ED56),QWord($A335F9DE920E0456),QWord($176E632D59157144),
QWord($37B8036FBDE47E04),QWord($8DB69C255B58351D),QWord($75E3AD18FA15C435),
QWord($B88F8FA79E7ACD2E),QWord($F4F1ADB22C44430A),QWord($A6F667B7787DE551),
QWord($7F5A7205AAB32994),QWord($415894996FFDADD7),QWord($0066BB6CDDC58D69),
QWord($73355DBCC89631CF),QWord($4431FD499E8875DB),QWord($497D7F92A785C0A1),
QWord($9E1259957CD82824),QWord($06AA4F3FE70CA2A2),QWord($2089374C701AD9E5),
QWord($9232A152F12FA6F5),QWord($8642D5B1E978751D),QWord($B983D466746FC062),
QWord($B927B4CDD4A40163),QWord($D318EFA72D4E8E2C),QWord($0739C2B135066E85),
QWord($288D2F5EC554BB59),QWord($E932090E1594B11D),QWord($99AAE93439757411),
QWord($B6355EBE7352D0D5),QWord($C55036D033D4CE1A),QWord($CF62E455765770ED),
QWord($D35B32D0879CEC65),QWord($4A7F75A91E55A387),QWord($DBA77CCE6BBA86C5),
QWord($A5BD40453312CC34),QWord($C873CF4AD2B21BDA),QWord($FB57ACBBFFF82A86),
QWord($548532941E5C97F5),QWord($C66A0BF8E8A5EBA2),QWord($4263CE77F7EB6DE5),
QWord($2E8A6099FB9CB50C),QWord($1D45AB6111B0DCB0),QWord($7FE52FF300874786),
QWord($595B7D86EA04E95F),QWord($02670CBA727270DC),QWord($DA2647DCAC755BCC),
QWord($587200296A36C48A),QWord($DCB2C01C8B852F55),QWord($9437511867ECA888),
QWord($DA15A6EBE12F1ACA),QWord($6BB01516C79BA56C),QWord($59D5512650DA86FE),
QWord($2508641E00EFB3D8),QWord($1082864880F8737B),QWord($B353426A76C9620F),
QWord($1133CD70565B0C38),QWord($4758F74C3AE9A9C7),QWord($3C3158E934173FD8),
QWord($932EAB7FF21856F6),QWord($EE6E49FA9C5D22BB),QWord($F0D5647DA187773A),
QWord($12D8F8D9175D50B5),QWord($CC1D89F26CA5F76B),QWord($AD350118288382EF),
QWord($2E8209A98951707A),QWord($429AC0E06E896040),QWord($1FA39EAB38C767F8),
QWord($0DB3875C9089C512),QWord($D833C218B2296004),QWord($51D022375837CDA6),
QWord($B197EFD3310610EC),QWord($96D9426A03DD99F1),QWord($A668A96A49BD5595),
QWord($88B9C5A482DEF3CF),QWord($CE29AA1123CE0CCE),QWord($E8C5D1BEDFEB8F4C),
QWord($9FD5C941FA43F815),QWord($71403B035037F8D6),QWord($F96985771B6CAEF5),
QWord($B0CEAEF1407FA726),QWord($DAE80323BC5687EE),QWord($E766D91350438AB1),
QWord($83352D6317D820D4),QWord($471FAF84D19B9D6B),QWord($B68B6E05C551787A),
QWord($0D82CB9A2FD0ED0A),QWord($9C9FA69B835533CF),QWord($529708C8B631A586),
QWord($AE6E7D9A9A021C8D),QWord($3776E16D46423ABD),QWord($E26768F257FA285A),
QWord($D1064D1D8071EC0B),QWord($87DD00CA296EC314),QWord($55664BAA77126B71),
QWord($84923CEA48B80A5D),QWord($CEEF0FB7CDBE63E8),QWord($EE9CD1F8F1BA69D1),
QWord($630CB18DCEF9F6D6),QWord($9ACCCFEC1E5B0B8B),QWord($A7C603D6C1771990),
QWord($8D8CFCA125F0FC18),QWord($21B3040B45983A58),QWord($779277D37F6F40B7),
QWord($E00E05C70D0DD723),QWord($2E06042A483503D1),QWord($11E6C7594E2D37BC),
QWord($B05774BB3125D878),QWord($F2F37B26D9686660),QWord($EC467268C4B86304),
QWord($BCB60695B6C2FE57),QWord($BD0F1F36040BE959),QWord($3717E742E40C6ACB),
QWord($3CC4804C944F7A20),QWord($460408E2E1B06095),QWord($A670D0945BE59F3A),
QWord($302057B370FF72B1),QWord($DA167250DCC3541C),QWord($05640270451078B4),
QWord($B76AA32013FC0D9A),QWord($46AEEECD80943D79),QWord($40ECDF130731B4A1),
QWord($97AD3D679928A042),QWord($504F5C189FA51C84),QWord($4E87AFD137482693),
QWord($A2BADFB0FAFEA200),QWord($1F87EA7EC1EC7232),QWord($5AE7467498550A62),
QWord($46078EAF639D8B94),QWord($FCD6039345449DC5),QWord($F1D0921E12BBF3E1),
QWord($A8278D27BF1494D9),QWord($3AAB6E1FDAA2078C),QWord($A7056D8748FE8393),
QWord($EDE587F7AA7F0D10),QWord($7CC2F685085F2B73),QWord($AB013AA88CD2F44F),
QWord($E728EE7494DB1A90),QWord($91E7AD1C7A9DC3A0),QWord($0B50F837CCCA0AB9),
QWord($6F4FAF4B3591C776),QWord($8ECEF7BB5EB62F62),QWord($38B856766C7E761F),
QWord($A804C793B28B69D6),QWord($E1ED19F49068CDB7),QWord($49104B4C4DD93914),
QWord($D5EAB4E714DBDCCE),QWord($C5CC85B5C060507F),QWord($DB32425C0D97E0AA),
QWord($40982C04911C28C2),QWord($2CC09E5BBEB228EB),QWord($12D0EE550AE54C7F),
QWord($CDA4DBE4290E1192)),

(
QWord($C8E7EE2A6D35A85D),QWord($E65C4BA2BDB172A3),QWord($24E778560394EA96),
QWord($C2B27A43E3B8DE3A),QWord($68851CA62A4E9602),QWord($5C55530FE3B65709),
QWord($879085274307A299),QWord($55EF07A5128D13E4),QWord($9D725027DBFCE0D1),
QWord($1F0A36B7042A8604),QWord($C0B998CFE7483209),QWord($68C3492A53196622),
QWord($74AB18A58026E986),QWord($B2A80581C9BB8E61),QWord($F6BEA5D51868B77E),
QWord($FB9EEF919206448B),QWord($D994BE0A28AB72B3),QWord($9E4F201F17E3A924),
QWord($E417494C2B40948B),QWord($F181F1AD48176EC8),QWord($4E6DFACE8A54167E),
QWord($03EAE7EBCCD76504),QWord($ADB580CF9AEFD7EC),QWord($C95F10516FC1822C),
QWord($16B4DBCA7E2AF7B6),QWord($4C2C1E92A3FD8B1E),QWord($2C47E98AF402DB8F),
QWord($41B35ECC6648A3F4),QWord($3E6304DBBAB7D72F),QWord($7FC1AAEAF908800E),
QWord($221D7A4945772D5B),QWord($2C2E0622FC7D0EBA),QWord($9642246B05B23630),
QWord($0A7B0F45486B606D),QWord($13177F6FCE892312),QWord($D1CB0F1F6DFCC606),
QWord($AA78E533377B4884),QWord($D6B52F4A8C9E2A60),QWord($99B73FD0ACE073AC),
QWord($6C15A0193AB9C716),QWord($DFC2B9DAB731F594),QWord($346570BDBEEFF775),
QWord($72C4AA3D7F315855),QWord($33A724C96D575675),QWord($048D4945B24864EB),
QWord($4A133231D8592CC0),QWord($D5C73DB05E524096),QWord($37FF9F3C0083DDB2),
QWord($62CF329FB4CA2C7A),QWord($FA32EB7340C80018),QWord($44A0BEBCC3FF0AF4),
QWord($E69FDA32090494F2),QWord($5BC6B8759CD4735D),QWord($EBD794C34F999AC3),
QWord($15EC4AB381966658),QWord($04227560AAD6E119),QWord($666F53B1C2D0E5CF),
QWord($FDA528678C635D9A),QWord($A95E2B06AF981F38),QWord($AC8D213BAF14763E),
QWord($618980C706184FD8),QWord($065FB48BFBED58F2),QWord($3530BF7072F45B25),
QWord($63496CF13DA894A7),QWord($10B0AAB9B53D00D5),QWord($C94D06690A504E9F),
QWord($B0188F98A1416C8F),QWord($D617690189DB2B89),QWord($D6C5973072B0D296),
QWord($41C5AF732550CA99),QWord($1F14A4CD472685D4),QWord($7FF0EB3059940A8F),
QWord($6AD4BB06B2CFC19A),QWord($26CAAFF27D31965B),QWord($24BAC753CA8C7495),
QWord($0BD00E001AD03531),QWord($C85A61F41B4F63BF),QWord($C069E56EA641FC8B),
QWord($58F03B9F82249453),QWord($3845C8E628849C2C),QWord($35A07EAA2B311D24),
QWord($BD87A6CAA8808EE4),QWord($6132969AF471FEEE),QWord($AD3CA83EFFC050F2),
QWord($92F28FADFE9548EE),QWord($4A445E95E2B96A7F),QWord($0CD740D338AC4E9E),
QWord($22CBDBA02CACEE06),QWord($2EF0B47ED4A26551),QWord($95162DFE6F77218E),
QWord($949BBE6E2C5E18CB),QWord($D858457AE86AEB8F),QWord($70160C4C32FCC035),
QWord($DA1B117537022CC1),QWord($4155B446242E9516),QWord($0EBB3A91C05F6DC1),
QWord($72CE4A139155F85E),QWord($968D1D2BE2202DAA),QWord($0DD112E69A43590B),
QWord($54571D7106AA1177),QWord($8185CF7C873A83D2),QWord($B60E081435FC7665),
QWord($1119443E61477BCD),QWord($BE341A447FBF3154),QWord($207BB1320C48C47F),
QWord($4ABDA6DE378E0293),QWord($6107CEF20273CE94),QWord($DB975B82DE2E74B7),
QWord($B5C328A45210420A),QWord($FD5DB064FF0CE588),QWord($A13D0310EA1451F2),
QWord($43B7383D053E52AE),QWord($0E8EF259EB10920A),QWord($AD3AD1E9745478B4),
QWord($6205B40C776A6AF2),QWord($555382797179A5A3),QWord($2FFE1524EFCC58F9),
QWord($7CF888C7BC0799F2),QWord($CDA43BA405DDBB75),QWord($B5477636677732F6),
QWord($01319EC2188D7305),QWord($C304C3C9C905CE9C),QWord($F1FE7A84086D9818),
QWord($33FFC90248AFEFEC),QWord($5F936E7E0B73F2A5),QWord($997533FE57D3D2BC),
QWord($A8727C21D3025478),QWord($B9A5ECB6D2B53213),QWord($A719EC619141CF64),
QWord($575191E04E65313A),QWord($E939E987FCC67DC8),QWord($E311DC20B4BF98CA),
QWord($83B18F3FE3C6B19A),QWord($45A3F7DBE43309F2),QWord($29C88527BB7E1798),
QWord($9BD9065395ED5092),QWord($3AFA47133CB87F2F),QWord($79D33156F2A0AEB5),
QWord($4E391928829ED724),QWord($45976E853A57156C),QWord($274ECB26774225DC),
QWord($F01D518D64D90A76),QWord($B7AA9CD93BDF7DFC),QWord($675187C1DDE860D2),
QWord($5B27441797399AA3),QWord($34A484AE25A7D8CC),QWord($3313743EEA047C38),
QWord($6642F851A39F0FE0),QWord($7E784F26458566B6),QWord($1667EB5A365039B7),
QWord($320CD5C287747C42),QWord($C500626988237C54),QWord($1900CD551BEA8F06),
QWord($1CB2EB6D1009C5F8),QWord($FE4467CE9DE6B84C),QWord($474907E7E930D4DE),
QWord($9A03748021094135),QWord($1D01A2CD1671E5E7),QWord($EA36E4D4BAEE3186),
QWord($FDC80A189B81415B),QWord($F05E5C2CEAD51307),QWord($B10BF4AB1BF091DD),
QWord($9A360064E7EFF780),QWord($002D8FD679033659),QWord($2CB6755A67B7F1A9),
QWord($74289C32DAC31036),QWord($8B43C5F7E46D7626),QWord($A5824FF7A1C04253),
QWord($BE299B6E486689D6),QWord($A6A36640B0E7ED10),QWord($EFCE1B3B69319579),
QWord($A4C7511EEF2E25D6),QWord($E734535A0A39E4D7),QWord($0D4058B97CBA88FF),
QWord($3FA7B14B24CDC6CD),QWord($97B32577F69EE7D2),QWord($F456EADE82A767FB),
QWord($0010AE7A6C3454D4),QWord($35A647CF2DFE414F),QWord($5F939DE1E6B3A564),
QWord($761CD4B5F4B8F9DE),QWord($32701D9F38859564),QWord($927008126BC94ED2),
QWord($04CA39A82D3EE4A0),QWord($26FF262BE4C1CD83),QWord($D8171EDC5EAF11B2),
QWord($E67378C44FB8DBBE),QWord($43821B229ED5B95D),QWord($F7049AAC3635CBE4),
QWord($1C4DDDB37B9AFD05),QWord($F0CBF5B4691CD3D3),QWord($6976C062ED0FFE99),
QWord($D936DB478E654D9B),QWord($22F34DBE8B13F2E7),QWord($CCB7DC2DE2731B06),
QWord($85F8BD4FCBDF97FA),QWord($02FAB7045B1C055F),QWord($97A94E31DB778B87),
QWord($2882DF5FC805577D),QWord($B695D74F9A30DE4A),QWord($0CF049C0A7345FA5),
QWord($96B9509F55F63483),QWord($3084D0AFBE224AAD),QWord($84CA453BC82105E9),
QWord($9B4D589D4A02FECF),QWord($3136FBBA8192321A),QWord($BD97D146DF9F4E2B),
QWord($B9F8040962D96DA1),QWord($EDCF51D37D9621F7),QWord($8DECA39042A16FFD),
QWord($9441F853C1504B57),QWord($9D1024FD82563051),QWord($D34746588B2CE81A),
QWord($6BEA6D58DF62C819),QWord($31029351D2E7904E),QWord($DDB03A88345FCBDB),
QWord($E62307DB8BADD207),QWord($32E5BB8925D326F4),QWord($290DDC6CF8B1C598),
QWord($0B7248827F06A000),QWord($7679EE16865E48EB),QWord($1F3BAB764D8FE8CF),
QWord($3CB0E848F54B3538),QWord($956D9C6EECEC3965),QWord($73EC71CE0828103A),
QWord($86FBEE17D8845ADD),QWord($E60D3AFF0F007CBE),QWord($89D3033A550F2A11),
QWord($2E081952F84293A0),QWord($974F0F245BD0A5E7),QWord($FE79914859C60354),
QWord($8982ACEAFD60F579),QWord($E1233F40905919BE),QWord($FAB1DFF38F021851),
QWord($C430D68AFFB03AA2),QWord($2BB72E416FAA9D5E),QWord($AD3A44FE8079644A),
QWord($2E9561FB8B801944),QWord($3300A1E2C054CCF4),QWord($D8084D497FFF7A10),
QWord($C33E8B2B8828E728),QWord($D1B3DEB329DD52C0),QWord($31B88B43F606C5FE),
QWord($B4212E76E6DC6095),QWord($93324E6411A911AA),QWord($5138BEA009DEBCCC),
QWord($5BBA38F1987EECB8),QWord($A2FCFCA27891EFB9),QWord($FB0C2EF580179636),
QWord($0625A4AE64B186C9),QWord($F3CF2FCBC3C58C0F),QWord($F5EA740F98982795),
QWord($2564A9C150FFBA8C),QWord($E62645E8043D11BF),QWord($BF3FCCC3619B3C35),
QWord($73D10F6B60D11815)),

(
QWord($A8E12082D7624114),QWord($9245F1A7A0997271),QWord($AD40A4311D954016),
QWord($C18B37D3FFBEA3CA),QWord($9F75CB789BDCC8D1),QWord($543457A71981F784),
QWord($7940B92E57823865),QWord($EE91834176D1A1CF),QWord($265FF2A579D6FAED),
QWord($B66BF86CB191E358),QWord($B0E5C8A0964F04AE),QWord($407D0DC8595D3934),
QWord($5DFF91E9D6352EC1),QWord($B9D495DD6FE7EBD8),QWord($815BCF0EDFEDB01F),
QWord($C02BF2BB2A2A6FA7),QWord($CD2F1CEA53958F27),QWord($0683592403762063),
QWord($0F26652B5091303B),QWord($E43FFB8885812210),QWord($3205ADA9F0D3640A),
QWord($22807158D37A7403),QWord($8B17A7AAA8121956),QWord($E5B239C95AE32EF1),
QWord($A14B442A7EF8C780),QWord($E1449CA4F16ADD48),QWord($E6079B9D7714A475),
QWord($9A9F36ADB6FCE01C),QWord($053444FDEF512622),QWord($9E1C5A27A97414F3),
QWord($97158027964BBA33),QWord($3577193E76F39380),QWord($503C38D0899EE5A7),
QWord($56F8A8386D84D297),QWord($46301A01246C0F8A),QWord($5340554DD1C86B9A),
QWord($5C02E7A5A846AEAD),QWord($94AF414D21FFE6CE),QWord($7AB72F4275B1701D),
QWord($37800AA8B54E391C),QWord($796EC8A820644CDB),QWord($3257B2C90337228A),
QWord($36FA9CEEE74E624B),QWord($8715D5B910D11736),QWord($F450DFB3840B3808),
QWord($877336D0A96226C1),QWord($F1B1D7EAF1EF9164),QWord($A234ABEB1CC6D6F5),
QWord($5C9378781B51529B),QWord($851E350D710845C6),QWord($5DD5164748FED079),
QWord($9327398D83D5CD98),QWord($B3721C8D87BDA627),QWord($21F6D79FC003D794),
QWord($3375B31563AB92B8),QWord($83BC7E579D936ECC),QWord($3DA3D1EB726E0F38),
QWord($21AEDA1E95BDA637),QWord($877FFA5529B29937),QWord($432303F07D4DD09E),
QWord($F4970D01770AD733),QWord($D9A755DAD173E41F),QWord($C8CE6C2632B896E0),
QWord($9C20F5D8623E9196),QWord($5B90DCC953048D23),QWord($AF339F2A14763FD0),
QWord($4BB23C18895CD2BA),QWord($8F9F3E95F22CEA61),QWord($374B5A39DB49BB9D),
QWord($E8D0649554DD3240),QWord($3686F79BD4AA41F0),QWord($92345E7B3F4B9805),
QWord($B9210C91B1B72896),QWord($909B40770A717BD4),QWord($7165196685C3BA29),
QWord($819E949DC1F68FF5),QWord($35981A79BDC9B23D),QWord($A8ABA317D6362F42),
QWord($4395E8D918C5D29E),QWord($24A9AF953ED557DC),QWord($1E8B609F9289CF48),
QWord($CF94872A5E21CC20),QWord($88F2A9DFFF3D5B30),QWord($F82694A79A525209),
QWord($A6730AD7996CB74D),QWord($5960CA440F7B3D2D),QWord($10272FCB8474D797),
QWord($11B5C7975713E5C5),QWord($881D6984156E7F67),QWord($4265E61A40384B34),
QWord($658F30B6F57BFE38),QWord($2D30CEC9B7D2141F),QWord($B4DE84C92F235B18),
QWord($F6C84891BBD361B1),QWord($09AC2B3F99A4DF60),QWord($139D9360FEE621FA),
QWord($35E1ACEFD867644F),QWord($95D6FE63E2C3ED5B),QWord($4127228476DAD854),
QWord($9A030F1D35D0610A),QWord($EF67FCA008316188),QWord($1AE36D97427BFC2B),
QWord($AA0A541207EBCDD1),QWord($F21A86A982BA0A5B),QWord($A158626CC13DA4F2),
QWord($8454855F71F362B5),QWord($A4A02DA6CD956970),QWord($3A1933A20A413562),
QWord($7FB08FD190C1750D),QWord($95C9D6D8F0A5597F),QWord($905F7A12F8B9F05F),
QWord($14F063982721D6C8),QWord($E1C81055EAF345E4),QWord($8B02B1F9A7B91569),
QWord($907307319BB4D3A2),QWord($22FC429A762E202B),QWord($4C8298D584C50557),
QWord($4EE680F9D55E8E1C),QWord($A59AADB37A64D4AB),QWord($777AEF54255C6BAF),
QWord($B9E9CD12C9F6D03F),QWord($0486405D89E12012),QWord($A8F32195AD9B848A),
QWord($24536A63053F1662),QWord($B8ABDF34B4352046),QWord($008E066DDF7B1BF0),
QWord($0937FA7D9D183D1A),QWord($1A5396EF432972AF),QWord($B07471DD5EA1912A),
QWord($FA5DFDA5DB809F3B),QWord($151E0E66B664D54F),QWord($1BBC2143B288B9AB),
QWord($ABC02F7631CC1927),QWord($F9A4AE96F9834DB0),QWord($F8193501A12927BA),
QWord($650FCD815770113E),QWord($E0CE618E5B52DA6D),QWord($B5B152BE30A8B07C),
QWord($DAD8F353972ECC52),QWord($99D4AEBC060A6A0F),QWord($08E62828868BB696),
QWord($5167F893396D2F36),QWord($1F2E9E73C80E8F85),QWord($E0AB823094E08AB3),
QWord($DC0BEABAACC05EDD),QWord($B6753A97DA5A8058),QWord($CFDB118A67D4916F),
QWord($51D9C89214F493F4),QWord($C26167305289B384),QWord($97160AB71A7E4CEF),
QWord($973179255169012B),QWord($7771F3EF6E0C219E),QWord($5E09D8BBE3A6AE29),
QWord($A711508CEC25DD3E),QWord($04279E53EFF42D64),QWord($9B8A1B3518BB56E9),
QWord($221111F31D86B8D4),QWord($E97D5EE7ABDE1FB8),QWord($BC734F44DF744FAC),
QWord($DFC032C98833DA84),QWord($F7A162F11B585ECC),QWord($A6D1166246CE8EEA),
QWord($0C6BD217156F8815),QWord($439D1F080BA22869),QWord($652E7C122E040C21),
QWord($1747E6D3B2983347),QWord($16B5B9143BDB4F64),QWord($0E17A5C775C2EE40),
QWord($53F23FAD5AC19C45),QWord($937348E6DECA163C),QWord($26D8F5211D908278),
QWord($2CA9D3196906EBAA),QWord($64D9C221B876DBE8),QWord($F5BD98B39765093A),
QWord($F2EFCBB39AC97411),QWord($A01C2699CADA8BBF),QWord($6ABC75F617DBEC7C),
QWord($23A862F326261E25),QWord($7DEF17B30E43CCF9),QWord($E454CB7EC51A1D90),
QWord($8D441891ABA6BEBE),QWord($2F2DA70DCC76EF77),QWord($439676424FA5A358),
QWord($49BBDCEE76067631),QWord($F63A6BD29775D777),QWord($5DFDDB9B12E8FCA9),
QWord($09A674E06451281A),QWord($E418486E062AC31D),QWord($3B180BBE4A1E515A),
QWord($D0CB5C961B63E731),QWord($9E8149929AD99C72),QWord($EC51E0A5193CA3EB),
QWord($FA21CE56A89D7E17),QWord($B08547231034F2EB),QWord($B0EEFD1E2BC1FBAA),
QWord($0A8AD2D2753F3F42),QWord($E283BEFA009FF8EC),QWord($E1D4FC1FF2132D83),
QWord($9DAE527A8DE56FF6),QWord($336F47D657E3EFB0),QWord($660AAEC7B4DA8111),
QWord($A6DCB60F391303AC),QWord($F85A3D5BF5D59398),QWord($900E22F664D6875E),
QWord($FA4C029F198DBAAD),QWord($4D773F305C40752E),QWord($EC2386ABD852D287),
QWord($7B1A582851235AEF),QWord($207FF1EA50D9D7E1),QWord($E969BE8B38C1865C),
QWord($99FA2391FD44F7B7),QWord($910EF30F8F61DACE),QWord($7E313EAFA58D460E),
QWord($CF3AC88D24E28A32),QWord($61B92C3C1990400C),QWord($940CE033C5F94165),
QWord($D035E3678AD93195),QWord($35CF0C9BA37347E2),QWord($7798C3C39EDF6978),
QWord($7FBFD530B558D69B),QWord($A6CF9EA4C6FD06A8),QWord($56E996F2C3C1629F),
QWord($0E1DC63B951C705C),QWord($CC124FBCD97D41DC),QWord($400342714ACDD337),
QWord($1365793B1B5E6497),QWord($869266F1299C7352),QWord($D74CEA89E67B7194),
QWord($C47E30EE84EF357F),QWord($0DB3483C5653CD54),QWord($8C9A962EB061CFE0),
QWord($BB242F2B4B07D64A),QWord($25814925B674FFAD),QWord($2ADFEB635DD35CC9),
QWord($AF510E642C60DA37),QWord($169475C9F7DC6F53),QWord($D4D553F641A01782),
QWord($E9FAEAEC2FFE94CE),QWord($4936FA1BBBD9097E),QWord($7B8A614E856107AB),
QWord($D21A345A686A03B4),QWord($50C20AC660123213),QWord($6997AFA323E3573D),
QWord($430C76E287C5D8F5),QWord($4BFE61429794D597),QWord($EDA7F3F70FF8F1A1),
QWord($0A137800E0F0F7E7),QWord($9C376D1E98B7BE7C),QWord($10F892DF487C4C27),
QWord($812AB00AB6486D79),QWord($24C3C23FA0CB549B),QWord($5F20AAC86BB637D8),
QWord($BF2FAD7CDA8F11B8),QWord($9B14B76D08EF72BE),QWord($8A0E0EEC190EBEBA),
QWord($8CF97F6ECE339C68))

);


function B_F(ctx : PBlowfish2Context; x: QWord) : QWord;

var
   a, b, c, d, e, f, g, h : cardinal;
   y : qword;
begin
   h:=Cardinal(x and $FF);
   x := x shr 8;
   g := Cardinal(x and $FF);
   x := x shr 8;
   f := Cardinal(x and $FF);
   x := x shr 8;
   e := Cardinal(x and $FF);
   x := x shr 8;
   d := Cardinal(x and $FF);
   x := x shr 8;
   c := Cardinal(x and $FF);
   x := x shr 8;
   b := Cardinal(x and $FF);
   x := x shr 8;
   a := Cardinal(x and $FF);

   y := QWord(ctx^.S[0][a] + ctx^.S[1][b]);
   y := QWord(y xor ctx^.S[2][c]);
   y := QWord(y + ctx^.S[3][d]);
   y := QWord(y xor ctx^.S[4][e]);
   y := QWord(y + ctx^.S[5][f]);
   y := QWord(y xor ctx^.S[6][g]);
   y := QWord(y + ctx^.S[7][h]);

   result:=y;
end;


procedure Blowfish2_Encrypt(ctx : PBlowfish2Context; var xl, xr: QWord);
var
  l, r, temp : QWord;
  i : smallint;
begin
  l := xl;
  r := xr;

  For i:= 0 to N-1 do
    begin
    l := QWord(l xor ctx^.P[i]);
    r := QWord(B_F(ctx, l) xor r);

    temp := l;
    l := r;
    r := temp;
    end;

  temp := l;
  l := r;
  r := temp;

  r := QWord(r xor ctx^.P[N]);
  l := QWord(l xor ctx^.P[N + 1]);

  xl := l;
  xr := r;
end;

procedure Blowfish2_Encrypt(ctx : PBlowfish2Context; var aBlock : TBF2Block); inline;

begin
  Blowfish2_Encrypt(ctx,aBlock[0],aBlock[1]);
end;

procedure Blowfish2_Decrypt(ctx : PBlowfish2Context; var xl, xr: QWord);

var
  l,r,temp : QWord;
  i : Smallint;

begin
  l := xl;
  r := xr;

  For I:=N + 1 downto 2 do
    begin
    l := QWord(l xor ctx^.P[i]);
    r := QWord(B_F(ctx, l) xor r);

    (* Exchange l and r *)
    temp := l;
    l := r;
    r := temp;
    end;

  (* Exchange Xl and Xr *)
  temp := l;
  l := r;
  r := temp;

  r := QWord(r xor ctx^.P[1]);
  l := QWord(l xor ctx^.P[0]);

  xl := l;
  xr := r;
end;

procedure Blowfish2_Decrypt(ctx : PBlowfish2Context; var aBlock : TBF2Block); inline;

begin
  Blowfish2_Decrypt(ctx,aBlock[0],aBlock[1]);
end;

procedure Blowfish2_Init(ctx : PBlowfish2Context; Key : TBytes);

begin
  BlowFish2_Init(ctx,PAnsiChar(Key),Length(Key));
end;

procedure Blowfish2_Init(ctx : PBlowfish2Context; key : pansichar; keyLen : Integer);

var
  i, j, k : longint;
  data, datal, datar : qword;

begin
  if KeyLen>MaxKeyLen then
    Raise EBlowFish2Error.CreateFmt(SErrKeyTooLong,[KeyLen,MaxKeyLen]);
  for i := 0 to 7 do
    for j:=0 to 255 do
      ctx^.S[i][j] := ORIG_S[i][j];
  j:=0;
  for i:=0 to N+1 do
    begin
    data := QWord($0000000000000000);
    for k:=0 to 7 do
      begin
      data := QWord((data shl 8) or Ord(key[j]));
      j := j + 1;
      if (j >= keyLen) then
        j := 0;
      end;
    ctx^.P[i] := QWord(ORIG_P[i] xor data);
    end;

  datal := QWord($0000000000000000);
  datar := QWord($0000000000000000);

  I:=0;
  While (I< N + 2) do
    begin
    Blowfish2_Encrypt(ctx, datal, datar);
    ctx^.P[i] := datal;
    ctx^.P[i + 1] := datar;
    inc(i, 2);
    end;

  for i:=0 to 7 do
    begin
    J:=0;
    While J<256 do
      begin
      Blowfish2_Encrypt(ctx, datal, datar);
      ctx^.S[i][j] := datal;
      ctx^.S[i][j + 1] := datar;
      inc(J,2);
      end
   end;
end;

{ ---------------------------------------------------------------------
    TBlowFish2Stream
  ---------------------------------------------------------------------}

function TBlowFish2Stream.GetPosition: Int64;
begin
  Result:=FPos;
end;

procedure TBlowFish2Stream.InvalidSeek;
begin
  raise EBlowFish2Error.Create(SNoSeekAllowed);
end;

constructor TBlowFish2Stream.Create(const AKey: UTF8String; Dest: TStream);

begin
  inherited Create(Dest);
  If Length(aKey)=0 then
    Raise EBlowFish2Error.Create(SErrEmptyPassPhraseNotAllowed);
  BlowFish2_init(@FBF,PAnsiChar(AKey),Length(aKey));
  FBufPos:=0;
  FPos:=0;
end;

constructor TBlowFish2Stream.Create(AKey: TBytes; Dest: TStream);

begin
  inherited Create(Dest);
  If Length(aKey)=0 then
    Raise EBlowFish2Error.Create(SErrEmptyPassPhraseNotAllowed);
  BlowFish2_init(@FBF,aKey);
  FBufPos:=0;
  FPos:=0;
end;

destructor TBlowFish2Stream.Destroy;

begin
  FreeAndNil(FBF);
  Inherited;
end;

{ ---------------------------------------------------------------------
    TBlowFish2EncryptStream
  ---------------------------------------------------------------------}

Destructor TBlowFish2EncryptStream.Destroy;


begin
  Flush;
  Inherited Destroy;
end;

Procedure TBlowFish2EncryptStream.Flush;

begin
  If FBufPos>0 then
    begin
    // Fill with nulls
    FillChar(PChar(@FData)[FBufPos],SizeOf(FData)-FBufPos,#0);
    Blowfish2_Encrypt(@FBF,FData);
    Source.Write(FData,SizeOf(FData));
    FBufPos := 0;
    end;
end;

function TBlowFish2EncryptStream.Write(const Buffer; Count: Longint): Longint;

Var
  mvsize : Longint;

begin
  Result:=0;
  While Count>0 do
    begin
    MVsize:=Count;
    If Mvsize>SizeOf(Fdata)-FBufPos then
      mvsize:=SizeOf(FData)-FBufPos;
    Move(PChar(@Buffer)[Result],PChar(@FData)[FBufPos],MVSize);
    If FBufPos+mvSize=Sizeof(FData) then
      begin
      // Empty buffer.
      Blowfish2_Encrypt(@FBF,FData);
      // this will raise an exception if needed.
      Source.Writebuffer(FData,SizeOf(FData));
      FBufPos:=0;
      end
    else
      inc(FBufPos,mvsize);
    Dec(Count,MvSize);
    Inc(Result,mvSize);
    end;
  Inc(FPos,Result);
end;


function TBlowFish2EncryptStream.Seek(const Offset: Int64; Origin: TSeekOrigin): Int64;

begin
  if (Offset = 0) and (Origin = soCurrent) then
    Result := FPos
  else
    InvalidSeek;
end;


{ ---------------------------------------------------------------------
    TBlowFishDecryptStream
  ---------------------------------------------------------------------}

Constructor TBlowFish2DeCryptStream.Create(AKey : TBytes; Dest: TStream); overload;

begin
  inherited Create(AKey,Dest);
  FSourcePos0 := Source.Position;
end;

Constructor TBlowFish2DeCryptStream.Create(const AKey : UTF8String; Dest: TStream); overload;

begin
  inherited Create(AKey, Dest);
  FSourcePos0 := Source.Position;
end;

function TBlowFish2DeCryptStream.Read(var Buffer; Count: Longint): Longint;

Var
  mvsize : Longint;

begin
  Result:=0;
  While Count>0 do
    begin
    // Empty existing buffer.
    If (FBufPos>0) then
      begin
      mvSize:=FBufPos;
      If MvSize>count then
        mvsize:=Count;
      Move(PChar(@FData)[0],PChar(@Buffer)[Result],MVSize);
      If ((Sizeof(FData)-MvSize)>0) then
        Move(PChar(@FData)[mvSize],PChar(@FData)[0],Sizeof(FData)-MvSize);
      Dec(Count,mvsize);
      Inc(Result,mvsize);
      FBufPos:=FBufPos-MvSize;
      end;
    // Fill buffer again if needed.
    If  (Count>0) then
      Begin
      mvsize:=Source.Read(FData,SizeOf(FData));
      If mvsize>0 then
        begin
        If MvSize<SizeOf(FData) Then
          // Fill with nulls
          FillChar(PChar(@FData)[mvsize],SizeOf(FData)-mvsize,#0);
        Blowfish2_Decrypt(@FBF,FData);
        FBufPos:=SizeOf(FData);
        end
      else
        Count:=0; // No more data available from stream; st
      end;
    end;
  Inc(FPos,Result);
end;

function TBlowFish2DeCryptStream.Seek(const Offset: Int64; Origin: TSeekOrigin): Int64;

begin
  if (Offset=0) and (Origin=soBeginning) then
  begin // support seek to beginning
    FBufPos:=0;
    FPos:=0;
    Source.Position := FSourcePos0;
  end else
    FakeSeekForward(Offset,TSeekOrigin(Origin),FPos);
  Result:=FPos;
end;


end.

