#!/usr/bin/env python3
import argparse
import glob
import os
import re

arg_parser = argparse.ArgumentParser(description="""CMake Quick Docs post-processing.""")
arg_parser.add_argument('--quickdoc_directory', type=str, required=True)
arg_parser.add_argument('--fix_sphinx_navigation_tags', action='store_true', required=False, help="Add a dummy comment to empty span tags with an id attribute to force anchor navigation working in CLion Quick Docs.")
arg_parser.add_argument('--fix_sphinx_headlines', action='store_true', required=False, help="Adds 'definintion' class attribute to h1 headline to fix CPP-37510")

def run_fix_sphinx_navigation_tags(quickdoc_dir):
  # empty span tag with id attribute
  attribute_pattern = '[a-zA-Z-_0-9]+="[^"]*"'
  id_attribute_pattern = 'id="[^"]*"'
  starttag_pattern = f'(?P<tag1><\s*span\s+({attribute_pattern}\s+)*{id_attribute_pattern}(\s+{attribute_pattern})*\s*>)'
  endtag_pattern = '(?P<tag2><\s*/span\s*>)'
  pattern = f'{starttag_pattern}{endtag_pattern}'

  # substitution
  dummy_comment = '<!--f-->'
  substitution = f'\g<tag1>{dummy_comment}\g<tag2>'

  # process CMake quick docs files
  html_paths = glob.glob(os.path.join(quickdoc_dir,"**", "*.html"))
  for html_path in html_paths:
    with open(html_path, 'r') as f:
      text = f.read()

    substituted_text = re.sub(pattern, substitution, text)

    with open(html_path, 'w') as f:
      f.write( substituted_text)

def run_fix_sphinx_headlines(quickdoc_dir):
  # h1 headline
  pattern = f'<h1>'

  # substitution
  substitution = f'<h1 class="definition">'

  # process CMake quick docs files
  html_paths = glob.glob(os.path.join(quickdoc_dir,"**", "*.html"))
  for html_path in html_paths:
    with open(html_path, 'r') as f:
      text = f.read()

    substituted_text = re.sub(pattern, substitution, text)

    with open(html_path, 'w') as f:
      f.write( substituted_text)


def main(args):
  if args.fix_sphinx_navigation_tags:
    run_fix_sphinx_navigation_tags(args.quickdoc_directory)
  if args.fix_sphinx_headlines:
    run_fix_sphinx_headlines(args.quickdoc_directory)

if __name__ == "__main__":
  main(arg_parser.parse_known_args()[0])
