/*
 * Decompiled with CFR 0.152.
 */
package com.android.internal.display;

import android.annotation.RequiresPermission;
import android.annotation.SuppressLint;
import android.content.ContentResolver;
import android.content.Context;
import android.database.ContentObserver;
import android.hardware.display.BrightnessInfo;
import android.hardware.display.DisplayManager;
import android.net.Uri;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.os.SystemClock;
import android.provider.Settings;
import android.ravenwood.annotation.RavenwoodKeep;
import android.ravenwood.annotation.RavenwoodKeepPartialClass;
import android.util.MathUtils;
import android.util.Slog;
import android.view.Display;
import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.display.BrightnessUtils;
import java.io.PrintWriter;

@RavenwoodKeepPartialClass
public class BrightnessSynchronizer {
    private static final String TAG = "BrightnessSynchronizer";
    private static final boolean DEBUG = false;
    private static final Uri BRIGHTNESS_URI = Settings.System.getUriFor("screen_brightness");
    private static final long WAIT_FOR_RESPONSE_MILLIS = 200L;
    private static final int MSG_RUN_UPDATE = 1;
    public static final float EPSILON = 1.0E-4f;
    private static int sBrightnessUpdateCount = 1;
    private final Context mContext;
    private final BrightnessSyncObserver mBrightnessSyncObserver;
    private final Clock mClock;
    private final Handler mHandler;
    private DisplayManager mDisplayManager;
    private int mLatestIntBrightness;
    private float mLatestFloatBrightness;
    private BrightnessUpdate mCurrentUpdate;
    private BrightnessUpdate mPendingUpdate;
    private final boolean mIntRangeUserPerceptionEnabled;

    public BrightnessSynchronizer(Context context, Looper looper, boolean intRangeUserPerceptionEnabled) {
        this(context, looper, SystemClock::uptimeMillis, intRangeUserPerceptionEnabled);
    }

    @VisibleForTesting
    public BrightnessSynchronizer(Context context, Looper looper, Clock clock, boolean intRangeUserPerceptionEnabled) {
        this.mContext = context;
        this.mClock = clock;
        this.mBrightnessSyncObserver = new BrightnessSyncObserver();
        this.mHandler = new BrightnessSynchronizerHandler(looper);
        this.mIntRangeUserPerceptionEnabled = intRangeUserPerceptionEnabled;
    }

    public void startSynchronizing() {
        if (this.mDisplayManager == null) {
            this.mDisplayManager = this.mContext.getSystemService(DisplayManager.class);
        }
        if (this.mBrightnessSyncObserver.isObserving()) {
            Slog.wtf(TAG, "Brightness sync observer requesting synchronization a second time.");
            return;
        }
        this.mLatestFloatBrightness = this.getScreenBrightnessFloat();
        this.mLatestIntBrightness = this.getScreenBrightnessInt();
        Slog.i(TAG, "Initial brightness readings: " + this.mLatestIntBrightness + "(int), " + this.mLatestFloatBrightness + "(float)");
        if (!Float.isNaN(this.mLatestFloatBrightness)) {
            this.mPendingUpdate = new BrightnessUpdate(2, this.mLatestFloatBrightness);
        } else if (this.mLatestIntBrightness != -1) {
            this.mPendingUpdate = new BrightnessUpdate(1, this.mLatestIntBrightness);
        } else {
            float defaultBrightness = this.mContext.getResources().getFloat(17105166);
            this.mPendingUpdate = new BrightnessUpdate(2, defaultBrightness);
            Slog.i(TAG, "Setting initial brightness to default value of: " + defaultBrightness);
        }
        this.mBrightnessSyncObserver.startObserving(this.mHandler);
        this.mHandler.sendEmptyMessageAtTime(1, this.mClock.uptimeMillis());
    }

    public void dump(PrintWriter pw) {
        pw.println("BrightnessSynchronizer:");
        pw.println("-----------------------");
        pw.println("  mLatestIntBrightness=" + this.mLatestIntBrightness);
        pw.println("  mLatestFloatBrightness=" + this.mLatestFloatBrightness);
        pw.println("  mCurrentUpdate=" + this.mCurrentUpdate);
        pw.println("  mPendingUpdate=" + this.mPendingUpdate);
        pw.println("  mIntRangeUserPerceptionEnabled=" + this.mIntRangeUserPerceptionEnabled);
    }

    public static float brightnessIntToFloat(int brightnessInt) {
        if (brightnessInt == 0) {
            return -1.0f;
        }
        if (brightnessInt == -1) {
            return Float.NaN;
        }
        float minFloat = 0.0f;
        float maxFloat = 1.0f;
        float minInt = 1.0f;
        float maxInt = 255.0f;
        return MathUtils.constrainedMap(0.0f, 1.0f, 1.0f, 255.0f, brightnessInt);
    }

    public static int brightnessFloatToInt(float brightnessFloat) {
        return Math.round(BrightnessSynchronizer.brightnessFloatToIntRange(brightnessFloat));
    }

    public static float brightnessFloatToIntRange(float brightnessFloat) {
        if (BrightnessSynchronizer.floatEquals(brightnessFloat, -1.0f)) {
            return 0.0f;
        }
        if (Float.isNaN(brightnessFloat)) {
            return -1.0f;
        }
        float minFloat = 0.0f;
        float maxFloat = 1.0f;
        float minInt = 1.0f;
        float maxInt = 255.0f;
        return MathUtils.constrainedMap(1.0f, 255.0f, 0.0f, 1.0f, brightnessFloat);
    }

    private void handleBrightnessChangeFloat(float brightness) {
        this.mLatestFloatBrightness = brightness;
        this.handleBrightnessChange(2, brightness);
    }

    private void handleBrightnessChangeInt(int brightness) {
        this.mLatestIntBrightness = brightness;
        this.handleBrightnessChange(1, brightness);
    }

    private void handleBrightnessChange(int type, float brightness) {
        boolean swallowUpdate = this.mCurrentUpdate != null && this.mCurrentUpdate.swallowUpdate(type, brightness);
        BrightnessUpdate prevUpdate = null;
        if (!swallowUpdate) {
            prevUpdate = this.mPendingUpdate;
            this.mPendingUpdate = new BrightnessUpdate(type, brightness);
        }
        this.runUpdate();
        if (!swallowUpdate && this.mPendingUpdate != null) {
            Slog.i(TAG, "New PendingUpdate: " + this.mPendingUpdate + ", prev=" + prevUpdate);
        }
    }

    private void runUpdate() {
        do {
            if (this.mCurrentUpdate != null) {
                this.mCurrentUpdate.update();
                if (this.mCurrentUpdate.isRunning()) break;
                if (this.mCurrentUpdate.isCompleted()) {
                    if (this.mCurrentUpdate.madeUpdates()) {
                        Slog.i(TAG, "Completed Update: " + this.mCurrentUpdate);
                    }
                    this.mCurrentUpdate = null;
                }
            }
            if (this.mCurrentUpdate != null || this.mPendingUpdate == null) continue;
            this.mCurrentUpdate = this.mPendingUpdate;
            this.mPendingUpdate = null;
        } while (this.mCurrentUpdate != null);
    }

    private float getScreenBrightnessFloat() {
        return this.mDisplayManager.getBrightness(0);
    }

    private int getScreenBrightnessInt() {
        return Settings.System.getIntForUser(this.mContext.getContentResolver(), "screen_brightness", -1, -2);
    }

    @RavenwoodKeep
    public static boolean floatEquals(float a, float b) {
        if (a == b) {
            return true;
        }
        if (Float.isNaN(a) && Float.isNaN(b)) {
            return true;
        }
        return Math.abs(a - b) < 1.0E-4f;
    }

    @RequiresPermission(value="android.permission.CONTROL_DISPLAY_BRIGHTNESS")
    public static float brightnessIntSettingToFloat(Context context, int brightnessInt) {
        if (brightnessInt == 0) {
            return -1.0f;
        }
        if (brightnessInt == -1) {
            return Float.NaN;
        }
        float minInt = 1.0f;
        float maxInt = 255.0f;
        float userPerceptionBrightness = MathUtils.norm(1.0f, 255.0f, brightnessInt);
        float linearBrightness = BrightnessUtils.convertGammaToLinear(userPerceptionBrightness);
        Display display = context.getDisplay();
        if (display == null) {
            return Float.NaN;
        }
        BrightnessInfo info = display.getBrightnessInfo();
        if (info == null) {
            return Float.NaN;
        }
        return MathUtils.lerp(info.brightnessMinimum, info.brightnessMaximum, linearBrightness);
    }

    @RequiresPermission(value="android.permission.CONTROL_DISPLAY_BRIGHTNESS")
    public static int brightnessFloatToIntSetting(Context context, float brightnessFloat) {
        if (BrightnessSynchronizer.floatEquals(brightnessFloat, -1.0f)) {
            return 0;
        }
        if (Float.isNaN(brightnessFloat)) {
            return -1;
        }
        Display display = context.getDisplay();
        if (display == null) {
            return -1;
        }
        BrightnessInfo info = display.getBrightnessInfo();
        if (info == null) {
            return -1;
        }
        float linearBrightness = MathUtils.norm(info.brightnessMinimum, info.brightnessMaximum, brightnessFloat);
        float userPerceptionBrightness = BrightnessUtils.convertLinearToGamma(linearBrightness);
        float minInt = 1.0f;
        float maxInt = 255.0f;
        float intBrightness = MathUtils.lerp(1.0f, 255.0f, userPerceptionBrightness);
        return Math.round(intBrightness);
    }

    @VisibleForTesting
    public static interface Clock {
        public long uptimeMillis();
    }

    private class BrightnessSyncObserver {
        private boolean mIsObserving;
        private final DisplayManager.DisplayListener mListener = new DisplayManager.DisplayListener(){

            @Override
            public void onDisplayAdded(int displayId) {
            }

            @Override
            public void onDisplayRemoved(int displayId) {
            }

            @Override
            public void onDisplayChanged(int displayId) {
                BrightnessSynchronizer.this.handleBrightnessChangeFloat(BrightnessSynchronizer.this.getScreenBrightnessFloat());
            }
        };

        private BrightnessSyncObserver() {
        }

        private ContentObserver createBrightnessContentObserver(Handler handler) {
            return new ContentObserver(handler){

                @Override
                public void onChange(boolean selfChange, Uri uri) {
                    if (selfChange) {
                        return;
                    }
                    if (BRIGHTNESS_URI.equals(uri)) {
                        BrightnessSynchronizer.this.handleBrightnessChangeInt(BrightnessSynchronizer.this.getScreenBrightnessInt());
                    }
                }
            };
        }

        boolean isObserving() {
            return this.mIsObserving;
        }

        void startObserving(Handler handler) {
            ContentResolver cr = BrightnessSynchronizer.this.mContext.getContentResolver();
            cr.registerContentObserver(BRIGHTNESS_URI, false, this.createBrightnessContentObserver(handler), -1);
            BrightnessSynchronizer.this.mDisplayManager.registerDisplayListener(this.mListener, handler, 0L, 1L);
            this.mIsObserving = true;
        }
    }

    class BrightnessSynchronizerHandler
    extends Handler {
        BrightnessSynchronizerHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            switch (msg.what) {
                case 1: {
                    BrightnessSynchronizer.this.runUpdate();
                    break;
                }
                default: {
                    super.handleMessage(msg);
                }
            }
        }
    }

    @VisibleForTesting
    public class BrightnessUpdate {
        static final int TYPE_INT = 1;
        static final int TYPE_FLOAT = 2;
        private static final int STATE_NOT_STARTED = 1;
        private static final int STATE_RUNNING = 2;
        private static final int STATE_COMPLETED = 3;
        private final int mSourceType;
        private final float mBrightness;
        private long mTimeUpdated;
        private int mState;
        private int mUpdatedTypes;
        private int mConfirmedTypes;
        private int mId = sBrightnessUpdateCount++;

        BrightnessUpdate(int sourceType, float brightness) {
            this.mSourceType = sourceType;
            this.mBrightness = brightness;
            this.mTimeUpdated = 0L;
            this.mUpdatedTypes = 0;
            this.mConfirmedTypes = 0;
            this.mState = 1;
        }

        public String toString() {
            return "{[" + this.mId + "] " + this.toStringLabel(this.mSourceType, this.mBrightness) + ", mUpdatedTypes=" + this.mUpdatedTypes + ", mConfirmedTypes=" + this.mConfirmedTypes + ", mTimeUpdated=" + this.mTimeUpdated + "}";
        }

        void update() {
            if (this.mState == 1) {
                float brightnessFloat;
                this.mState = 2;
                int brightnessInt = this.getBrightnessAsInt();
                if (BrightnessSynchronizer.this.mLatestIntBrightness != brightnessInt) {
                    Settings.System.putIntForUser(BrightnessSynchronizer.this.mContext.getContentResolver(), "screen_brightness", brightnessInt, -2);
                    BrightnessSynchronizer.this.mLatestIntBrightness = brightnessInt;
                    this.mUpdatedTypes |= 1;
                }
                if (!BrightnessSynchronizer.floatEquals(BrightnessSynchronizer.this.mLatestFloatBrightness, brightnessFloat = this.getBrightnessAsFloat())) {
                    BrightnessSynchronizer.this.mDisplayManager.setBrightness(0, brightnessFloat);
                    BrightnessSynchronizer.this.mLatestFloatBrightness = brightnessFloat;
                    this.mUpdatedTypes |= 2;
                }
                if (this.mUpdatedTypes != 0) {
                    Slog.i(BrightnessSynchronizer.TAG, "[" + this.mId + "] New Update " + this.toStringLabel(this.mSourceType, this.mBrightness) + " set brightness values: " + this.toStringLabel(this.mUpdatedTypes & 2, brightnessFloat) + " " + this.toStringLabel(this.mUpdatedTypes & 1, brightnessInt));
                    BrightnessSynchronizer.this.mHandler.sendEmptyMessageAtTime(1, BrightnessSynchronizer.this.mClock.uptimeMillis() + 200L);
                }
                this.mTimeUpdated = BrightnessSynchronizer.this.mClock.uptimeMillis();
            }
            if (this.mState == 2 && (this.mConfirmedTypes == this.mUpdatedTypes || this.mTimeUpdated + 200L < BrightnessSynchronizer.this.mClock.uptimeMillis())) {
                this.mState = 3;
            }
        }

        boolean swallowUpdate(int type, float brightness) {
            boolean intUpdateConfirmed;
            if ((this.mUpdatedTypes & type) != type || (this.mConfirmedTypes & type) != 0) {
                return false;
            }
            boolean floatUpdateConfirmed = type == 2 && BrightnessSynchronizer.floatEquals(this.getBrightnessAsFloat(), brightness);
            boolean bl = intUpdateConfirmed = type == 1 && this.getBrightnessAsInt() == (int)brightness;
            if (floatUpdateConfirmed || intUpdateConfirmed) {
                this.mConfirmedTypes |= type;
                Slog.i(BrightnessSynchronizer.TAG, "Swallowing update of " + this.toStringLabel(type, brightness) + " by update: " + this);
                return true;
            }
            return false;
        }

        boolean isRunning() {
            return this.mState == 2;
        }

        boolean isCompleted() {
            return this.mState == 3;
        }

        boolean madeUpdates() {
            return this.mUpdatedTypes != 0;
        }

        @SuppressLint(value={"AndroidFrameworkRequiresPermission"})
        private int getBrightnessAsInt() {
            if (this.mSourceType == 1) {
                return (int)this.mBrightness;
            }
            if (BrightnessSynchronizer.this.mIntRangeUserPerceptionEnabled) {
                return BrightnessSynchronizer.brightnessFloatToIntSetting(BrightnessSynchronizer.this.mContext, this.mBrightness);
            }
            return BrightnessSynchronizer.brightnessFloatToInt(this.mBrightness);
        }

        @SuppressLint(value={"AndroidFrameworkRequiresPermission"})
        private float getBrightnessAsFloat() {
            if (this.mSourceType == 2) {
                return this.mBrightness;
            }
            if (BrightnessSynchronizer.this.mIntRangeUserPerceptionEnabled) {
                return BrightnessSynchronizer.brightnessIntSettingToFloat(BrightnessSynchronizer.this.mContext, (int)this.mBrightness);
            }
            return BrightnessSynchronizer.brightnessIntToFloat((int)this.mBrightness);
        }

        private String toStringLabel(int type, float brightness) {
            return type == 1 ? (int)brightness + "(i)" : (type == 2 ? brightness + "(f)" : "");
        }
    }
}

