/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.network;

import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.ScatteringByteChannel;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.kafka.common.memory.MemoryPool;
import org.apache.kafka.common.network.InvalidReceiveException;
import org.apache.kafka.common.network.Receive;
import org.apache.kafka.common.network.SslUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetworkReceive
implements Receive {
    public static final String UNKNOWN_SOURCE = "";
    public static final int UNLIMITED = -1;
    private static final Logger log = LoggerFactory.getLogger(NetworkReceive.class);
    private static final ByteBuffer EMPTY_BUFFER = ByteBuffer.allocate(0);
    private final String source;
    private final ByteBuffer sizeBuf;
    private final ByteBuffer minBuf;
    private final int maxSize;
    private final MemoryPool memoryPool;
    private final AtomicInteger byteCount;
    private int requestedBufferSize = -1;
    private ByteBuffer payloadBuffer = null;
    private volatile ReadState readState = ReadState.READ_SIZE;

    public NetworkReceive() {
        this(UNKNOWN_SOURCE);
    }

    public NetworkReceive(String source) {
        this(-1, source);
    }

    public NetworkReceive(String source, ByteBuffer buffer) {
        this(source);
        this.payloadBuffer = buffer;
    }

    public NetworkReceive(int maxSize, String source) {
        this(maxSize, source, MemoryPool.NONE);
    }

    public NetworkReceive(int maxSize, String source, MemoryPool memoryPool) {
        this.source = source;
        this.maxSize = maxSize;
        this.memoryPool = memoryPool;
        this.minBuf = (ByteBuffer)ByteBuffer.allocate(5).position(4);
        this.sizeBuf = (ByteBuffer)this.minBuf.duplicate().position(0).limit(4);
        this.byteCount = new AtomicInteger(0);
    }

    @Override
    public long readFrom(ScatteringByteChannel channel) throws IOException {
        int read = 0;
        switch (this.readState) {
            case READ_SIZE: {
                read += this.readRequestedBufferSize(channel);
                if (this.sizeBuf.hasRemaining()) break;
                this.readState = ReadState.VALIDATE_SIZE;
            }
            case VALIDATE_SIZE: {
                if (this.requestedBufferSize != 0) {
                    read += this.validateRequestedBufferSize(channel);
                    if (this.minBuf.hasRemaining()) break;
                }
                this.readState = ReadState.ALLOCATE_BUFFER;
            }
            case ALLOCATE_BUFFER: {
                if (this.requestedBufferSize == 0) {
                    this.payloadBuffer = EMPTY_BUFFER;
                } else {
                    this.payloadBuffer = this.tryAllocateBuffer(this.requestedBufferSize);
                    if (this.payloadBuffer == null) break;
                    this.minBuf.position(this.sizeBuf.limit());
                    this.payloadBuffer.put(this.minBuf);
                }
                this.readState = ReadState.READ_PAYLOAD;
            }
            case READ_PAYLOAD: {
                int payloadRead = channel.read(this.payloadBuffer);
                if (payloadRead < 0) {
                    throw new EOFException();
                }
                read += payloadRead;
                if (this.payloadBuffer.hasRemaining()) break;
                this.readState = ReadState.COMPLETE;
                break;
            }
        }
        this.byteCount.addAndGet(read);
        return read;
    }

    private int validateRequestedBufferSize(ScatteringByteChannel channel) throws IOException {
        int minRead = channel.read(this.minBuf);
        if (minRead < 0) {
            throw new EOFException();
        }
        if (!this.minBuf.hasRemaining()) {
            boolean isEncrypted = SslUtils.isEncrypted((ByteBuffer)this.minBuf.duplicate().rewind());
            if (isEncrypted) {
                throw new InvalidReceiveException("Recieved an unexpected SSL packet from the server. Please ensure the client is properly configured with SSL enabled.");
            }
            if (this.requestedBufferSize < 0) {
                throw new InvalidReceiveException("Invalid receive (size = " + this.requestedBufferSize + ")");
            }
            if (this.maxSize != -1 && this.requestedBufferSize > this.maxSize) {
                throw new InvalidReceiveException("Invalid receive (size = " + this.requestedBufferSize + " larger than " + this.maxSize + ")");
            }
        }
        return minRead;
    }

    private ByteBuffer tryAllocateBuffer(int bufSize) {
        ByteBuffer bb = this.memoryPool.tryAllocate(bufSize);
        if (bb == null) {
            log.trace("Broker low on memory - could not allocate buffer of size {} for source {}", (Object)this.requestedBufferSize, (Object)this.source);
        }
        return bb;
    }

    private int readRequestedBufferSize(ReadableByteChannel channel) throws IOException {
        int sizeRead = channel.read(this.sizeBuf);
        if (sizeRead < 0) {
            throw new EOFException();
        }
        if (this.sizeBuf.hasRemaining()) {
            return sizeRead;
        }
        this.sizeBuf.rewind();
        this.requestedBufferSize = this.sizeBuf.getInt();
        return sizeRead;
    }

    @Override
    public boolean requiredMemoryAmountKnown() {
        return this.readState.ordinal() > ReadState.VALIDATE_SIZE.ordinal();
    }

    @Override
    public boolean memoryAllocated() {
        return this.readState.ordinal() >= ReadState.READ_PAYLOAD.ordinal();
    }

    @Override
    public boolean complete() {
        return this.readState == ReadState.COMPLETE;
    }

    @Override
    public void close() throws IOException {
        if (this.payloadBuffer != null && this.payloadBuffer != EMPTY_BUFFER) {
            this.memoryPool.release(this.payloadBuffer);
            this.payloadBuffer = null;
        }
    }

    @Override
    public String source() {
        return this.source;
    }

    public ByteBuffer payload() {
        return this.payloadBuffer;
    }

    public int bytesRead() {
        return this.byteCount.get();
    }

    public int size() {
        return this.payload().limit() + this.sizeBuf.limit();
    }

    static enum ReadState {
        READ_SIZE,
        VALIDATE_SIZE,
        ALLOCATE_BUFFER,
        READ_PAYLOAD,
        COMPLETE;

    }
}

